/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
import { buildPhraseFilter } from './phrase_filter';
import { buildPhrasesFilter } from './phrases_filter';
import { buildRangeFilter } from './range_filter';
import { buildExistsFilter } from './exists_filter';

/**
 *
 * @param indexPattern
 * @param field
 * @param type
 * @param negate whether the filter is negated (NOT filter)
 * @param disabled  whether the filter is disabled andwon't be applied to searches
 * @param params
 * @param alias a display name for the filter
 * @param store whether the filter applies to the current application or should be applied to global context
 * @returns
 *
 * @public
 */
export function buildFilter(indexPattern, field, type, negate, disabled, params, alias, store) {
  var filter = buildBaseFilter(indexPattern, field, type, params);
  filter.meta.alias = alias;
  filter.meta.negate = negate;
  filter.meta.disabled = disabled;

  if (store) {
    filter.$state = {
      store: store
    };
  }

  return filter;
}

function buildBaseFilter(indexPattern, field, type, params) {
  switch (type) {
    case 'phrase':
      return buildPhraseFilter(field, params, indexPattern);

    case 'phrases':
      return buildPhrasesFilter(field, params, indexPattern);

    case 'range':
      var _ref = params,
          gte = _ref.from,
          lt = _ref.to;
      return buildRangeFilter(field, {
        lt: lt,
        gte: gte
      }, indexPattern);

    case 'range_from_value':
      return buildRangeFilter(field, params, indexPattern);

    case 'exists':
      return buildExistsFilter(field, indexPattern);

    default:
      throw new Error("Unknown filter type: ".concat(type));
  }
}