/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
export var makeRowManager = function makeRowManager(containerRef) {
  var rowIdToElements = new Map();
  return {
    getRow: function getRow(_ref) {
      var rowIndex = _ref.rowIndex,
          visibleRowIndex = _ref.visibleRowIndex,
          top = _ref.top,
          height = _ref.height;
      var rowElement = rowIdToElements.get(rowIndex);

      if (rowElement == null) {
        rowElement = document.createElement('div');
        rowElement.setAttribute('role', 'row');
        rowElement.dataset.gridRowIndex = String(rowIndex); // Row index from data, affected by sorting/pagination

        rowElement.dataset.gridVisibleRowIndex = String(visibleRowIndex); // Affected by sorting/pagination

        rowElement.classList.add('euiDataGridRow');
        var isOddRow = visibleRowIndex % 2 !== 0;
        if (isOddRow) rowElement.classList.add('euiDataGridRow--striped');
        rowElement.style.position = 'absolute';
        rowElement.style.left = '0';
        rowElement.style.right = '0'; // In order for the rowElement's left and right position to correctly inherit
        // from the innerGrid width, we need to make its position relative

        containerRef.current.style.position = 'relative'; // add the element to the wrapping container

        containerRef.current.appendChild(rowElement); // add the element to the row map

        rowIdToElements.set(rowIndex, rowElement); // watch the row's children, if they all disappear then remove this row

        var observer = new MutationObserver(function (records) {
          if (records[0].target.childElementCount === 0) {
            var _rowElement;

            observer.disconnect();
            (_rowElement = rowElement) === null || _rowElement === void 0 ? void 0 : _rowElement.remove();
            rowIdToElements.delete(rowIndex);
          }
        });
        observer.observe(rowElement, {
          childList: true
        });
      } // Ensure that the row's dimensions are always correct by having each cell update position styles


      rowElement.style.top = top;
      rowElement.style.height = "".concat(height, "px");
      return rowElement;
    }
  };
};