"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.remove = exports.listCommands = exports.list = exports.insertIntoCommand = exports.insertExpression = exports.insertCommand = exports.getCommand = exports.findByPredicate = exports.find = void 0;
var _builder = require("../../../builder");
var _visitor = require("../../../visitor");
var util = _interopRequireWildcard(require("../../util"));
var generic = _interopRequireWildcard(require("../../generic"));
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * This "template" allows the developer to easily specify a new sort expression
 * AST node, for example:
 *
 * ```ts
 * // as a simple string
 * 'column_name'
 *
 * // column with nested fields
 * ['column_name', 'nested_field']
 *
 * // as an object with additional options
 * { parts: 'column_name', order: 'ASC', nulls: 'NULLS FIRST' }
 * { parts: ['column_name', 'nested_field'], order: 'DESC', nulls: 'NULLS LAST' }
 * ```
 */

const createSortExpression = template => {
  var _template$order, _template$nulls;
  const parts = typeof template === 'string' ? [template] : Array.isArray(template) ? template : typeof template.parts === 'string' ? [template.parts] : template.parts;
  const identifiers = parts.map(part => _builder.Builder.identifier({
    name: part
  }));
  const column = _builder.Builder.expression.column({
    args: identifiers
  });
  if (typeof template === 'string' || Array.isArray(template)) {
    return column;
  }
  const order = _builder.Builder.expression.order(column, {
    order: (_template$order = template.order) !== null && _template$order !== void 0 ? _template$order : '',
    nulls: (_template$nulls = template.nulls) !== null && _template$nulls !== void 0 ? _template$nulls : ''
  });
  return order;
};

/**
 * Iterates through all sort commands starting from the beginning of the query.
 * You can specify the `skip` parameter to skip a given number of sort commands.
 *
 * @param ast The root of the AST.
 * @param skip Number of sort commands to skip.
 * @returns Iterator through all sort commands.
 */
const listCommands = (ast, skip = 0) => {
  return new _visitor.Visitor().on('visitSortCommand', function* (ctx) {
    if (skip) {
      skip--;
    } else {
      yield ctx.node;
    }
  }).on('visitCommand', function* () {}).on('visitQuery', function* (ctx) {
    for (const command of ctx.visitCommands()) {
      yield* command;
    }
  }).visitQuery(ast);
};

/**
 * Returns the Nth SORT command found in the query.
 *
 * @param ast The root of the AST.
 * @param index The index (N) of the sort command to return.
 * @returns The sort command found in the AST, if any.
 */
exports.listCommands = listCommands;
const getCommand = (ast, index = 0) => {
  for (const command of listCommands(ast, index)) {
    return command;
  }
};

/**
 * Returns an iterator for all sort expressions (columns and order expressions)
 * in the query. You can specify the `skip` parameter to skip a given number of
 * expressions.
 *
 * @param ast The root of the AST.
 * @param skip Number of sort expressions to skip.
 * @returns Iterator through sort expressions (columns and order expressions).
 */
exports.getCommand = getCommand;
const list = (ast, skip = 0) => {
  return new _visitor.Visitor().on('visitSortCommand', function* (ctx) {
    for (const argument of ctx.arguments()) {
      if (argument.type === 'order' || argument.type === 'column') {
        if (skip) {
          skip--;
        } else {
          yield [argument, ctx.node];
        }
      }
    }
  }).on('visitCommand', function* () {}).on('visitQuery', function* (ctx) {
    for (const command of ctx.visitCommands()) {
      yield* command;
    }
  }).visitQuery(ast);
};

/**
 * Finds the Nts sort expression that matches the predicate.
 *
 * @param ast The root of the AST.
 * @param predicate A function that returns true if the sort expression matches
 *     the predicate.
 * @param index The index of the sort expression to return. If not specified,
 *     the first sort expression that matches the predicate will be returned.
 * @returns The sort expressions and sort command 2-tuple that matches the
 *     predicate, if any.
 */
exports.list = list;
const findByPredicate = (ast, predicate, index) => {
  return util.findByPredicate(list(ast, index), predicate);
};

/**
 * Finds the Nth sort expression that matches the sort expression by column
 * name. The `parts` argument allows to specify an array of nested field names.
 *
 * @param ast The root of the AST.
 * @param parts A string or an array of strings representing the column name.
 * @returns The sort expressions and sort command 2-tuple that matches the
 *     predicate, if any.
 */
exports.findByPredicate = findByPredicate;
const find = (ast, parts, index = 0) => {
  const arrParts = typeof parts === 'string' ? [parts] : parts;
  return findByPredicate(ast, ([node]) => {
    let isMatch = false;
    if (node.type === 'column') {
      isMatch = util.cmpArr(node.args.map(arg => arg.type === 'identifier' ? arg.name : ''), arrParts);
    } else if (node.type === 'order') {
      var _node$args$;
      const columnParts = (_node$args$ = node.args[0]) === null || _node$args$ === void 0 ? void 0 : _node$args$.args;
      if (Array.isArray(columnParts)) {
        isMatch = util.cmpArr(columnParts.map(arg => arg.type === 'identifier' ? arg.name : ''), arrParts);
      }
    }
    if (isMatch) {
      index--;
      if (index < 0) {
        return true;
      }
    }
    return false;
  });
};

/**
 * Removes the Nth sort expression that matches the sort expression by column
 * name. The `parts` argument allows to specify an array of nested field names.
 *
 * @param ast The root of the AST.
 * @param parts A string or an array of strings representing the column name.
 * @param index The index of the sort expression to remove.
 * @returns The sort expressions and sort command 2-tuple that was removed, if any.
 */
exports.find = find;
const remove = (ast, parts, index) => {
  const tuple = find(ast, parts, index);
  if (!tuple) {
    return undefined;
  }
  const [node] = tuple;
  const cmd = generic.commands.args.remove(ast, node);
  if (cmd) {
    if (!cmd.args.length) {
      generic.commands.remove(ast, cmd);
    }
  }
  return cmd ? tuple : undefined;
};

/**
 * Inserts a new sort expression into the specified SORT command at the
 * specified argument position.
 *
 * @param sortCommand The SORT command to insert the new sort expression into.
 * @param template The sort expression template.
 * @param index Argument position in the command argument list.
 * @returns The inserted sort expression.
 */
exports.remove = remove;
const insertIntoCommand = (sortCommand, template, index) => {
  const expression = createSortExpression(template);
  generic.commands.args.insert(sortCommand, expression, index);
  return expression;
};

/**
 * Creates a new sort expression node and inserts it into the specified SORT
 * command at the specified argument position. If not sort command is found, a
 * new one is created and appended to the end of the query.
 *
 * @param ast The root AST node.
 * @param parts ES|QL column name parts.
 * @param index The new column name position in command argument list.
 * @param sortCommandIndex The index of the SORT command in the AST. E.g. 0 is the
 *     first SORT command in the AST.
 * @returns The inserted column AST node.
 */
exports.insertIntoCommand = insertIntoCommand;
const insertExpression = (ast, template, index = -1, sortCommandIndex = 0) => {
  let command = getCommand(ast, sortCommandIndex);
  if (!command) {
    command = _builder.Builder.command({
      name: 'sort'
    });
    generic.commands.append(ast, command);
  }
  return insertIntoCommand(command, template, index);
};

/**
 * Inserts a new SORT command with a single sort expression as its sole argument.
 * You can specify the position to insert the command at.
 *
 * @param ast The root of the AST.
 * @param template The sort expression template.
 * @param index The position to insert the sort expression at.
 * @returns The inserted sort expression and the command it was inserted into.
 */
exports.insertExpression = insertExpression;
const insertCommand = (ast, template, index = -1) => {
  const expression = createSortExpression(template);
  const command = _builder.Builder.command({
    name: 'sort',
    args: [expression]
  });
  generic.commands.insert(ast, command, index);
  return [command, expression];
};
exports.insertCommand = insertCommand;