"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CrawlCustomSettingsFlyoutMultiCrawlLogic = void 0;
var _kea = require("kea");
var _flash_messages = require("../../../../../shared/flash_messages");
var _http = require("../../../../../shared/http");
var _utils = require("../../../../api/crawler/utils");
var _index_name_logic = require("../../index_name_logic");
var _index_view_logic = require("../../index_view_logic");
var _crawl_custom_settings_flyout_domain_logic = require("./crawl_custom_settings_flyout_domain_logic");
var _crawl_custom_settings_flyout_logic = require("./crawl_custom_settings_flyout_logic");
var _crawl_custom_settings_flyout_schedule_api_logic = require("./crawl_custom_settings_flyout_schedule_api_logic");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const defaulCrawlerConfiguration = {
  scheduleKey: 'crawler_0',
  name: 'Crawler 0',
  maxCrawlDepth: 2,
  customEntryPointUrls: [],
  customSitemapUrls: [],
  includeSitemapsInRobotsTxt: true,
  selectedDomainUrls: [],
  selectedEntryPointUrls: [],
  selectedSitemapUrls: [],
  interval: '0 0 0 * * ?',
  enabled: false,
  sitemapUrls: [],
  entryPointUrls: []
};
const CrawlCustomSettingsFlyoutMultiCrawlLogic = exports.CrawlCustomSettingsFlyoutMultiCrawlLogic = (0, _kea.kea)({
  path: ['enterprise_search', 'crawler', 'crawl_custom_settings_flyout_multi_crawl_logic'],
  connect: {
    actions: [_crawl_custom_settings_flyout_schedule_api_logic.PostCustomSchedulingApiLogic, ['makeRequest as makePostCustomSchedulingRequest'], _crawl_custom_settings_flyout_domain_logic.CrawlCustomSettingsFlyoutDomainConfigLogic, ['onRecieveDomainConfigData']],
    values: [_index_view_logic.IndexViewLogic, ['index'], _crawl_custom_settings_flyout_domain_logic.CrawlCustomSettingsFlyoutDomainConfigLogic, ['domainConfigs', 'domainConfigMap']]
  },
  actions: () => ({
    fetchCustomScheduling: true,
    postCustomScheduling: true,
    onAddCustomCrawler: index => ({
      index
    }),
    onDeleteCustomCrawler: index => ({
      index
    }),
    onReceiveCrawlerCustomScheduling: crawlerConfigurations => ({
      crawlerConfigurations
    }),
    onSelectCrawlerConfigActiveTab: crawlerConfigActiveTab => ({
      crawlerConfigActiveTab
    }),
    onSelectCustomEntryPointUrls: (index, entryPointUrls) => ({
      index,
      entryPointUrls
    }),
    onSelectCustomSitemapUrls: (index, sitemapUrls) => ({
      index,
      sitemapUrls
    }),
    onSelectDomainUrls: (index, domainUrls) => ({
      index,
      domainUrls
    }),
    onSelectEntryPointUrls: (index, entryPointUrls) => ({
      index,
      entryPointUrls
    }),
    onSelectMaxCrawlDepth: (index, maxCrawlDepth) => ({
      index,
      maxCrawlDepth
    }),
    onSelectSitemapUrls: (index, sitemapUrls) => ({
      index,
      sitemapUrls
    }),
    onSetConnectorSchedulingEnabled: (index, enabled) => ({
      index,
      enabled
    }),
    setConnectorSchedulingInterval: (index, newSchedule) => ({
      index,
      newSchedule
    }),
    toggleIncludeSitemapsInRobotsTxt: index => ({
      index
    })
  }),
  reducers: () => ({
    crawlerConfigActiveTab: [0, {
      onSelectCrawlerConfigActiveTab: (_, {
        crawlerConfigActiveTab
      }) => crawlerConfigActiveTab,
      onDeleteCustomCrawler: () => 0
    }],
    crawlerConfigurations: [[defaulCrawlerConfiguration], {
      onReceiveCrawlerCustomScheduling: (_, {
        crawlerConfigurations
      }) => {
        // Handle case with no custom scheduling returned from server
        return crawlerConfigurations.length > 0 ? crawlerConfigurations.map(configuration => ({
          ...defaulCrawlerConfiguration,
          ...configuration
        })) : [defaulCrawlerConfiguration];
      },
      onAddCustomCrawler: (state, {
        index
      }) => {
        let newScheduleKey = `crawler_${index}`;
        let suffix = index;

        // Check if the newScheduleKey already exists in the array
        const existingKeys = state.map(crawler => crawler.scheduleKey);
        if (existingKeys.includes(newScheduleKey)) {
          // Handle the case where a duplicate scheduleKey is found
          while (existingKeys.includes(`${newScheduleKey}_${suffix}`)) {
            suffix++;
          }
          newScheduleKey = `${newScheduleKey}_${suffix}`;
        }
        return [...state, {
          ...defaulCrawlerConfiguration,
          name: `Crawler ${suffix}`,
          scheduleKey: newScheduleKey
        }];
      },
      onDeleteCustomCrawler: (state, {
        index
      }) => {
        return state.filter((_, i) => i !== index);
      },
      onSelectMaxCrawlDepth: (state, {
        index,
        maxCrawlDepth
      }) => {
        return state.map((crawler, i) => i === index ? {
          ...crawler,
          maxCrawlDepth
        } : crawler);
      },
      onSelectCustomEntryPointUrls: (state, {
        index,
        entryPointUrls
      }) => {
        return state.map((crawler, i) => i === index ? {
          ...crawler,
          customEntryPointUrls: entryPointUrls
        } : crawler);
      },
      onSelectCustomSitemapUrls: (state, {
        index,
        sitemapUrls
      }) => {
        return state.map((crawler, i) => i === index ? {
          ...crawler,
          customSitemapUrls: sitemapUrls
        } : crawler);
      },
      toggleIncludeSitemapsInRobotsTxt: (state, {
        index
      }) => {
        return state.map((crawler, i) => i === index ? {
          ...crawler,
          includeSitemapsInRobotsTxt: !crawler.includeSitemapsInRobotsTxt
        } : crawler);
      },
      onSelectDomainUrls: (state, {
        index,
        domainUrls
      }) => {
        return state.map((crawler, i) => i === index ? {
          ...crawler,
          selectedDomainUrls: domainUrls,
          selectedEntryPointUrls: (0, _crawl_custom_settings_flyout_logic.filterSeedUrlsByDomainUrls)(crawler.selectedEntryPointUrls, domainUrls),
          selectedSitemapUrls: (0, _crawl_custom_settings_flyout_logic.filterSeedUrlsByDomainUrls)(crawler.selectedSitemapUrls, domainUrls)
        } : crawler);
      },
      onSelectEntryPointUrls: (state, {
        index,
        entryPointUrls
      }) => {
        return state.map((crawler, i) => i === index ? {
          ...crawler,
          selectedEntryPointUrls: entryPointUrls
        } : crawler);
      },
      onSelectSitemapUrls: (state, {
        index,
        sitemapUrls
      }) => {
        return state.map((crawler, i) => i === index ? {
          ...crawler,
          selectedSitemapUrls: sitemapUrls
        } : crawler);
      },
      onSetConnectorSchedulingEnabled: (state, {
        index,
        enabled
      }) => {
        return state.map((crawler, i) => i === index ? {
          ...crawler,
          enabled
        } : crawler);
      },
      setConnectorSchedulingInterval: (state, {
        index,
        newSchedule
      }) => {
        const {
          interval
        } = newSchedule;
        return state.map((crawler, i) => i === index ? {
          ...crawler,
          interval
        } : crawler);
      },
      onRecieveDomainConfigData: (state, {
        domainConfigs
      }) => {
        const domainConfigsMap = (0, _crawl_custom_settings_flyout_domain_logic.domainConfigsToDomainConfigMap)(domainConfigs);
        return state.map(crawler => {
          const entryPointUrls = crawler.selectedDomainUrls.flatMap(selectedDomainUrl => domainConfigsMap[selectedDomainUrl].seedUrls);
          const selectedEntryPointUrls = crawler.customEntryPointUrls.filter(entryPointUrl => entryPointUrls.includes(entryPointUrl));
          const customEntryPointUrls = crawler.customEntryPointUrls.filter(entryPointUrl => !entryPointUrls.includes(entryPointUrl));
          const sitemapUrls = crawler.selectedDomainUrls.flatMap(selectedDomainUrl => domainConfigsMap[selectedDomainUrl].sitemapUrls);
          const selectedSitemapUrls = crawler.customSitemapUrls.filter(sitemapUrl => sitemapUrls.includes(sitemapUrl));
          const customSitemapUrls = crawler.customSitemapUrls.filter(sitemapUrl => !sitemapUrls.includes(sitemapUrl));
          return {
            ...crawler,
            entryPointUrls,
            selectedEntryPointUrls,
            customEntryPointUrls,
            sitemapUrls,
            selectedSitemapUrls,
            customSitemapUrls
          };
        });
      }
    }]
  }),
  selectors: () => ({
    crawlerConfigurationsWithDomainData: [selectors => [selectors.domainConfigMap, selectors.crawlerConfigurations], (domainConfigMap, crawlerConfigs) => crawlerConfigs.map(crawlerConfig => {
      const entryPointUrls = crawlerConfig.selectedDomainUrls.flatMap(selectedDomainUrl => domainConfigMap[selectedDomainUrl].seedUrls);
      const sitemapUrls = crawlerConfig.selectedDomainUrls.flatMap(selectedDomainUrl => domainConfigMap[selectedDomainUrl].sitemapUrls);
      return {
        ...crawlerConfig,
        entryPointUrls,
        sitemapUrls
      };
    })],
    crawlerCustomSchedulingIsValid: [selectors => [selectors.crawlerConfigurations], crawlerConfigs => crawlerConfigs.every(config => config.selectedDomainUrls.length > 0)]
  }),
  listeners: ({
    actions,
    values
  }) => ({
    fetchCustomScheduling: async () => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        indexName
      } = _index_name_logic.IndexNameLogic.values;
      try {
        const customSchedulingResponse = await http.get(`/internal/enterprise_search/indices/${indexName}/crawler/custom_scheduling`);
        const customScheduling = (0, _utils.crawlerCustomSchedulingServerToClient)(customSchedulingResponse);
        actions.onReceiveCrawlerCustomScheduling(customScheduling);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    postCustomScheduling: async () => {
      const {
        indexName
      } = _index_name_logic.IndexNameLogic.values;
      const {
        crawlerConfigurations
      } = values;
      const customScheduling = (0, _utils.crawlerCustomSchedulingClientToServer)(crawlerConfigurations);
      try {
        actions.makePostCustomSchedulingRequest({
          indexName,
          customScheduling
        });
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    }
  })
});