"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SynonymsLogic = void 0;
var _kea = require("kea");
var _flash_messages = require("../../../shared/flash_messages");
var _http = require("../../../shared/http");
var _table_pagination = require("../../../shared/table_pagination");
var _engine = require("../engine");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SynonymsLogic = exports.SynonymsLogic = (0, _kea.kea)({
  path: ['enterprise_search', 'app_search', 'synonyms_logic'],
  actions: () => ({
    loadSynonyms: true,
    onSynonymsLoad: ({
      results,
      meta
    }) => ({
      results,
      meta
    }),
    onPaginate: newPageIndex => ({
      newPageIndex
    }),
    openModal: synonymSet => ({
      synonymSet
    }),
    closeModal: true,
    createSynonymSet: synonyms => ({
      synonyms
    }),
    updateSynonymSet: ({
      id,
      synonyms
    }) => ({
      id,
      synonyms
    }),
    deleteSynonymSet: id => ({
      id
    }),
    onSynonymSetSuccess: successMessage => ({
      successMessage
    }),
    onSynonymSetError: true
  }),
  reducers: () => ({
    dataLoading: [true, {
      loadSynonyms: () => true,
      onSynonymsLoad: () => false
    }],
    synonymSets: [[], {
      onSynonymsLoad: (_, {
        results
      }) => results
    }],
    meta: [_constants.SYNONYMS_PAGE_META, {
      onSynonymsLoad: (_, {
        meta
      }) => meta,
      onPaginate: (state, {
        newPageIndex
      }) => (0, _table_pagination.updateMetaPageIndex)(state, newPageIndex)
    }],
    isModalOpen: [false, {
      openModal: () => true,
      closeModal: () => false
    }],
    activeSynonymSet: [null, {
      openModal: (_, {
        synonymSet
      }) => synonymSet,
      closeModal: () => null
    }],
    modalLoading: [false, {
      createSynonymSet: () => true,
      updateSynonymSet: () => true,
      deleteSynonymSet: () => true,
      onSynonymSetError: () => false,
      closeModal: () => false
    }]
  }),
  listeners: ({
    actions,
    values
  }) => ({
    loadSynonyms: async () => {
      const {
        meta
      } = values;
      const {
        http
      } = _http.HttpLogic.values;
      const {
        engineName
      } = _engine.EngineLogic.values;
      try {
        const response = await http.get(`/internal/app_search/engines/${engineName}/synonyms`, {
          query: {
            'page[current]': meta.page.current,
            'page[size]': meta.page.size
          }
        });
        actions.onSynonymsLoad(response);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    createSynonymSet: async ({
      synonyms
    }) => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        engineName
      } = _engine.EngineLogic.values;
      (0, _flash_messages.clearFlashMessages)();
      try {
        await http.post(`/internal/app_search/engines/${engineName}/synonyms`, {
          body: JSON.stringify({
            synonyms
          })
        });
        actions.onSynonymSetSuccess(_constants.CREATE_SUCCESS);
      } catch (e) {
        actions.onSynonymSetError();
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    updateSynonymSet: async ({
      id,
      synonyms
    }) => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        engineName
      } = _engine.EngineLogic.values;
      (0, _flash_messages.clearFlashMessages)();
      try {
        await http.put(`/internal/app_search/engines/${engineName}/synonyms/${id}`, {
          body: JSON.stringify({
            synonyms
          })
        });
        actions.onSynonymSetSuccess(_constants.UPDATE_SUCCESS);
      } catch (e) {
        actions.onSynonymSetError();
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    deleteSynonymSet: async ({
      id
    }) => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        engineName
      } = _engine.EngineLogic.values;
      (0, _flash_messages.clearFlashMessages)();
      try {
        await http.delete(`/internal/app_search/engines/${engineName}/synonyms/${id}`);
        actions.onSynonymSetSuccess(_constants.DELETE_SUCCESS);
      } catch (e) {
        actions.onSynonymSetError();
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    onSynonymSetSuccess: async ({
      successMessage
    }) => {
      await actions.loadSynonyms();
      actions.closeModal();
      (0, _flash_messages.flashSuccessToast)(successMessage, {
        text: _constants.SYNONYM_IMPACT_MESSAGE
      });
    }
  })
});