"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DefendInsightsDataClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _uuid = require("uuid");
var _ = require("..");
var _get_defend_insight = require("./get_defend_insight");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_PAGE_SIZE = 10;
class DefendInsightsDataClient extends _.AIAssistantDataClient {
  constructor(options) {
    super(options);
    /**
     * Fetches a Defend insight
     * @param options
     * @param options.id The existing Defend insight id.
     * @param options.authenticatedUser Current authenticated user.
     * @returns The Defend insight response
     */
    (0, _defineProperty2.default)(this, "getDefendInsight", async ({
      id,
      authenticatedUser
    }) => {
      const esClient = await this.options.elasticsearchClientPromise;
      return (0, _get_defend_insight.getDefendInsight)({
        esClient,
        logger: this.options.logger,
        index: this.indexTemplateAndPattern.alias,
        id,
        user: authenticatedUser
      });
    });
    /**
     * Creates a Defend insight, if given at least the "apiConfig"
     * @param options
     * @param options.defendInsightCreate
     * @param options.authenticatedUser
     * @returns The Defend insight created
     */
    (0, _defineProperty2.default)(this, "createDefendInsight", async ({
      defendInsightCreate,
      authenticatedUser
    }) => {
      const esClient = await this.options.elasticsearchClientPromise;
      const logger = this.options.logger;
      const index = this.indexTemplateAndPattern.alias;
      const user = authenticatedUser;
      const id = (defendInsightCreate === null || defendInsightCreate === void 0 ? void 0 : defendInsightCreate.id) || (0, _uuid.v4)();
      const createdAt = new Date().toISOString();
      const body = (0, _helpers.transformToCreateScheme)(createdAt, this.spaceId, user, defendInsightCreate);
      try {
        const response = await esClient.create({
          body,
          id,
          index,
          refresh: 'wait_for'
        });
        const createdDefendInsight = await (0, _get_defend_insight.getDefendInsight)({
          esClient,
          index,
          id: response._id,
          logger,
          user
        });
        return createdDefendInsight;
      } catch (err) {
        logger.error(`error creating Defend insight: ${err} with id: ${id}`);
        throw err;
      }
    });
    /**
     * Find Defend insights by params
     * @param options
     * @param options.params
     * @param options.authenticatedUser
     * @returns The Defend insights found
     */
    (0, _defineProperty2.default)(this, "findDefendInsightsByParams", async ({
      params,
      authenticatedUser
    }) => {
      const esClient = await this.options.elasticsearchClientPromise;
      const logger = this.options.logger;
      const index = this.indexTemplateAndPattern.alias;
      const user = authenticatedUser;
      const termFilters = (0, _helpers.queryParamsToEsQuery)(params);
      const filterByUser = [{
        nested: {
          path: 'users',
          query: {
            bool: {
              must: [{
                match: user.profile_uid ? {
                  'users.id': user.profile_uid
                } : {
                  'users.name': user.username
                }
              }]
            }
          }
        }
      }];
      try {
        const query = {
          bool: {
            must: [...termFilters, ...filterByUser]
          }
        };
        const response = await esClient.search({
          query,
          _source: true,
          ignore_unavailable: true,
          index,
          seq_no_primary_term: true,
          sort: [{
            '@timestamp': 'desc'
          }],
          size: params.size || DEFAULT_PAGE_SIZE
        });
        return (0, _helpers.transformESSearchToDefendInsights)(response);
      } catch (err) {
        logger.error(`error fetching Defend insights: ${err} with params: ${JSON.stringify(params)}`);
        throw err;
      }
    });
    /**
     * Finds all Defend insight for authenticated user
     * @param options
     * @param options.authenticatedUser
     * @returns The Defend insight
     */
    (0, _defineProperty2.default)(this, "findAllDefendInsights", async ({
      authenticatedUser
    }) => {
      const esClient = await this.options.elasticsearchClientPromise;
      const logger = this.options.logger;
      const index = this.indexTemplateAndPattern.alias;
      const user = authenticatedUser;
      const MAX_ITEMS = 10000;
      const filterByUser = [{
        nested: {
          path: 'users',
          query: {
            bool: {
              must: [{
                match: user.profile_uid ? {
                  'users.id': user.profile_uid
                } : {
                  'users.name': user.username
                }
              }]
            }
          }
        }
      }];
      try {
        const response = await esClient.search({
          query: {
            bool: {
              must: [...filterByUser]
            }
          },
          size: MAX_ITEMS,
          _source: true,
          ignore_unavailable: true,
          index,
          seq_no_primary_term: true
        });
        const insights = (0, _helpers.transformESSearchToDefendInsights)(response);
        return insights !== null && insights !== void 0 ? insights : [];
      } catch (err) {
        logger.error(`error fetching Defend insights: ${err}`);
        throw err;
      }
    });
    /**
     * Updates Defend insights
     * @param options
     * @param options.defendInsightsUpdateProps
     * @param options.authenticatedUser
     */
    (0, _defineProperty2.default)(this, "updateDefendInsights", async ({
      defendInsightsUpdateProps,
      authenticatedUser
    }) => {
      const esClient = await this.options.elasticsearchClientPromise;
      const logger = this.options.logger;
      const updatedAt = new Date().toISOString();
      let ids = [];
      const bulkParams = defendInsightsUpdateProps.flatMap(updateProp => {
        const index = updateProp.backingIndex;
        const params = (0, _helpers.transformToUpdateScheme)(updatedAt, updateProp);
        ids = [...ids, params.id];
        return [{
          update: {
            _index: index,
            _id: params.id
          }
        }, {
          doc: params
        }];
      });
      try {
        await esClient.bulk({
          body: bulkParams,
          refresh: 'wait_for'
        });
        return this.findDefendInsightsByParams({
          params: {
            ids
          },
          authenticatedUser
        });
      } catch (err) {
        logger.warn(`error updating Defend insights: ${err} for IDs: ${ids}`);
        throw err;
      }
    });
    /**
     * Updates a Defend insight
     * @param options
     * @param options.defendInsightUpdateProps
     * @param options.authenticatedUser
     */
    (0, _defineProperty2.default)(this, "updateDefendInsight", async ({
      defendInsightUpdateProps,
      authenticatedUser
    }) => {
      return (await this.updateDefendInsights({
        defendInsightsUpdateProps: [defendInsightUpdateProps],
        authenticatedUser
      }))[0];
    });
    this.options = options;
  }
}
exports.DefendInsightsDataClient = DefendInsightsDataClient;