"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateConversation = exports.transformToUpdateScheme = void 0;
var _helpers = require("./helpers");
var _transforms = require("./transforms");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const updateConversation = async ({
  conversationUpdateProps,
  dataWriter,
  logger,
  user
}) => {
  var _transformESToConvers;
  const updatedAt = new Date().toISOString();
  const params = transformToUpdateScheme(updatedAt, conversationUpdateProps);
  const {
    errors,
    docs_updated: docsUpdated
  } = await dataWriter.bulk({
    documentsToUpdate: [params],
    getUpdateScript: document => (0, _helpers.getUpdateScript)({
      conversation: document
    }),
    authenticatedUser: user
  });
  if (errors && errors.length > 0) {
    logger.warn(`Error updating conversation: ${errors.map(err => err.message)} by ID: ${params.id}`);
    return null;
  }
  const updatedConversation = (_transformESToConvers = (0, _transforms.transformESToConversations)(docsUpdated)) === null || _transformESToConvers === void 0 ? void 0 : _transformESToConvers[0];
  return updatedConversation;
};
exports.updateConversation = updateConversation;
const transformToUpdateScheme = (updatedAt, {
  title,
  apiConfig,
  excludeFromLastConversationStorage,
  messages,
  replacements,
  id
}) => {
  return {
    id,
    updated_at: updatedAt,
    ...(title ? {
      title
    } : {}),
    ...(apiConfig ? {
      api_config: {
        action_type_id: apiConfig === null || apiConfig === void 0 ? void 0 : apiConfig.actionTypeId,
        connector_id: apiConfig === null || apiConfig === void 0 ? void 0 : apiConfig.connectorId,
        default_system_prompt_id: apiConfig === null || apiConfig === void 0 ? void 0 : apiConfig.defaultSystemPromptId,
        model: apiConfig === null || apiConfig === void 0 ? void 0 : apiConfig.model,
        provider: apiConfig === null || apiConfig === void 0 ? void 0 : apiConfig.provider
      }
    } : {}),
    ...(excludeFromLastConversationStorage != null ? {
      exclude_from_last_conversation_storage: excludeFromLastConversationStorage
    } : {}),
    ...(replacements ? {
      replacements: Object.keys(replacements).map(key => ({
        uuid: key,
        value: replacements[key]
      }))
    } : {}),
    ...(messages ? {
      messages: messages.map(message => ({
        '@timestamp': message.timestamp,
        content: message.content,
        is_error: message.isError,
        reader: message.reader,
        role: message.role,
        ...(message.metadata ? {
          metadata: {
            ...(message.metadata.contentReferences ? {
              content_references: message.metadata.contentReferences
            } : {})
          }
        } : {}),
        ...(message.traceData ? {
          trace_data: {
            trace_id: message.traceData.traceId,
            transaction_id: message.traceData.transactionId
          }
        } : {})
      }))
    } : {})
  };
};
exports.transformToUpdateScheme = transformToUpdateScheme;