"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AutoOpsAPIService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _https = _interopRequireDefault(require("https"));
var _serverHttpTools = require("@kbn/server-http-tools");
var _elasticApmNode = _interopRequireDefault(require("elastic-apm-node"));
var _axios = _interopRequireDefault(require("axios"));
var _utils = require("../../common/utils");
var _rest_types = require("../../common/rest_types");
var _errors = require("../errors");
var _app_context = require("./app_context");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const AUTO_OPS_REQUEST_FAILED_PREFIX = '[AutoOps API] Request failed';
const AUTO_OPS_MISSING_CONFIG_ERROR = 'Missing autoops configuration';
const getAutoOpsAPIRequestUrl = (url, projectId) => `${url}/monitoring/serverless/v1/projects/${projectId}/metrics`;
class AutoOpsAPIService {
  constructor(logger) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "convertCauseErrorsToString", error => {
      if (error.cause instanceof AggregateError) {
        return error.cause.errors.map(e => e.message);
      }
      return error.cause;
    });
    this.logger = logger;
  }
  async autoOpsUsageMetricsAPI(requestBody) {
    var _apm$currentTransacti, _appContextService$ge, _autoopsConfig$api, _autoopsConfig$api2, _autoopsConfig$api2$t, _autoopsConfig$api3, _autoopsConfig$api3$t, _autoopsConfig$api4, _momentDateParser, _momentDateParser2;
    const traceId = (_apm$currentTransacti = _elasticApmNode.default.currentTransaction) === null || _apm$currentTransacti === void 0 ? void 0 : _apm$currentTransacti.traceparent;
    const withRequestIdMessage = message => `${message} [Request Id: ${traceId}]`;
    const errorMetadata = {
      trace: {
        id: traceId
      }
    };
    const autoopsConfig = (_appContextService$ge = _app_context.appContextService.getConfig()) === null || _appContextService$ge === void 0 ? void 0 : _appContextService$ge.autoops;
    if (!autoopsConfig) {
      this.logger.error(`[AutoOps API] ${AUTO_OPS_MISSING_CONFIG_ERROR}`, errorMetadata);
      throw new _errors.AutoOpsError(AUTO_OPS_MISSING_CONFIG_ERROR);
    }
    if (!((_autoopsConfig$api = autoopsConfig.api) !== null && _autoopsConfig$api !== void 0 && _autoopsConfig$api.url)) {
      this.logger.error(`[AutoOps API] Missing API URL in the configuration.`, errorMetadata);
      throw new _errors.AutoOpsError('Missing API URL in AutoOps configuration.');
    }
    if (!((_autoopsConfig$api2 = autoopsConfig.api) !== null && _autoopsConfig$api2 !== void 0 && (_autoopsConfig$api2$t = _autoopsConfig$api2.tls) !== null && _autoopsConfig$api2$t !== void 0 && _autoopsConfig$api2$t.certificate) || !((_autoopsConfig$api3 = autoopsConfig.api) !== null && _autoopsConfig$api3 !== void 0 && (_autoopsConfig$api3$t = _autoopsConfig$api3.tls) !== null && _autoopsConfig$api3$t !== void 0 && _autoopsConfig$api3$t.key)) {
      this.logger.error(`[AutoOps API] Missing required TLS certificate or key in the configuration.`, errorMetadata);
      throw new _errors.AutoOpsError('Missing required TLS certificate or key in AutoOps configuration.');
    }
    this.logger.debug(`[AutoOps API] Creating autoops agent with request URL: ${autoopsConfig.api.url} and TLS cert: [REDACTED] and TLS key: [REDACTED]`);
    const controller = new AbortController();
    const tlsConfig = this.createTlsConfig(autoopsConfig);
    const cloudSetup = _app_context.appContextService.getCloud();
    const requestConfig = {
      url: getAutoOpsAPIRequestUrl((_autoopsConfig$api4 = autoopsConfig.api) === null || _autoopsConfig$api4 === void 0 ? void 0 : _autoopsConfig$api4.url, cloudSetup === null || cloudSetup === void 0 ? void 0 : cloudSetup.serverless.projectId),
      data: {
        from: (_momentDateParser = (0, _utils.momentDateParser)(requestBody.from)) === null || _momentDateParser === void 0 ? void 0 : _momentDateParser.toISOString(),
        to: (_momentDateParser2 = (0, _utils.momentDateParser)(requestBody.to)) === null || _momentDateParser2 === void 0 ? void 0 : _momentDateParser2.toISOString(),
        size: requestBody.dataStreams.length,
        level: 'datastream',
        metric_types: requestBody.metricTypes,
        allowed_indices: requestBody.dataStreams
      },
      signal: controller.signal,
      method: 'POST',
      headers: {
        'Content-type': 'application/json',
        'X-Request-ID': traceId,
        traceparent: traceId
      },
      httpsAgent: new _https.default.Agent({
        rejectUnauthorized: tlsConfig.rejectUnauthorized,
        cert: tlsConfig.certificate,
        key: tlsConfig.key
      })
    };
    if (!(cloudSetup !== null && cloudSetup !== void 0 && cloudSetup.isServerlessEnabled)) {
      requestConfig.data.stack_version = _app_context.appContextService.getKibanaVersion();
    }
    const requestConfigDebugStatus = this.createRequestConfigDebug(requestConfig);
    this.logger.debug(`[AutoOps API] Creating autoops agent with request config ${requestConfigDebugStatus}`);
    const errorMetadataWithRequestConfig = {
      ...errorMetadata,
      http: {
        request: {
          id: traceId,
          body: requestConfig.data
        }
      }
    };
    const response = await (0, _axios.default)(requestConfig).catch(error => {
      if (!_axios.default.isAxiosError(error)) {
        this.logger.error(`${AUTO_OPS_REQUEST_FAILED_PREFIX} with an error ${error}, request config: ${requestConfigDebugStatus}`, errorMetadataWithRequestConfig);
        throw new _errors.AutoOpsError(withRequestIdMessage(error.message));
      }
      const errorLogCodeCause = `${error.code}  ${this.convertCauseErrorsToString(error)}`;
      if (error.response) {
        // The request was made and the server responded with a status code and error data
        this.logger.error(`${AUTO_OPS_REQUEST_FAILED_PREFIX} because the AutoOps API responded with a status code that falls out of the range of 2xx: ${JSON.stringify(error.response.status)}} ${JSON.stringify(error.response.data)}}, request config: ${requestConfigDebugStatus}`, {
          ...errorMetadataWithRequestConfig,
          http: {
            ...errorMetadataWithRequestConfig.http,
            response: {
              status_code: error.response.status,
              body: error.response.data
            }
          }
        });
        throw new _errors.AutoOpsError(withRequestIdMessage(AUTO_OPS_REQUEST_FAILED_PREFIX));
      } else if (error.request) {
        // The request was made but no response was received
        this.logger.error(`${AUTO_OPS_REQUEST_FAILED_PREFIX} while sending the request to the AutoOps API: ${errorLogCodeCause}, request config: ${requestConfigDebugStatus}`, errorMetadataWithRequestConfig);
        throw new _errors.AutoOpsError(withRequestIdMessage(`no response received from the AutoOps API`));
      } else {
        // Something happened in setting up the request that triggered an Error
        this.logger.error(`${AUTO_OPS_REQUEST_FAILED_PREFIX} with ${errorLogCodeCause}, request config: ${requestConfigDebugStatus}, error: ${error.toJSON()}`, errorMetadataWithRequestConfig);
        throw new _errors.AutoOpsError(withRequestIdMessage(`${AUTO_OPS_REQUEST_FAILED_PREFIX}, ${error.message}`));
      }
    });
    const validatedResponse = _rest_types.UsageMetricsAutoOpsResponseSchema.body().validate(response.data);
    this.logger.debug(`[AutoOps API] Successfully created an autoops agent ${response}`);
    return validatedResponse;
  }
  createTlsConfig(autoopsConfig) {
    var _autoopsConfig$api5, _autoopsConfig$api5$t, _autoopsConfig$api6, _autoopsConfig$api6$t;
    return new _serverHttpTools.SslConfig(_serverHttpTools.sslSchema.validate({
      enabled: true,
      certificate: autoopsConfig === null || autoopsConfig === void 0 ? void 0 : (_autoopsConfig$api5 = autoopsConfig.api) === null || _autoopsConfig$api5 === void 0 ? void 0 : (_autoopsConfig$api5$t = _autoopsConfig$api5.tls) === null || _autoopsConfig$api5$t === void 0 ? void 0 : _autoopsConfig$api5$t.certificate,
      key: autoopsConfig === null || autoopsConfig === void 0 ? void 0 : (_autoopsConfig$api6 = autoopsConfig.api) === null || _autoopsConfig$api6 === void 0 ? void 0 : (_autoopsConfig$api6$t = _autoopsConfig$api6.tls) === null || _autoopsConfig$api6$t === void 0 ? void 0 : _autoopsConfig$api6$t.key
    }));
  }
  createRequestConfigDebug(requestConfig) {
    return JSON.stringify({
      ...requestConfig,
      data: {
        ...requestConfig.data,
        fleet_token: '[REDACTED]'
      },
      httpsAgent: {
        ...requestConfig.httpsAgent,
        options: {
          ...requestConfig.httpsAgent.options,
          cert: requestConfig.httpsAgent.options.cert ? 'REDACTED' : undefined,
          key: requestConfig.httpsAgent.options.key ? 'REDACTED' : undefined
        }
      }
    });
  }
}
exports.AutoOpsAPIService = AutoOpsAPIService;