"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createRoot = createRoot;
exports.createRootWithCorePlugins = createRootWithCorePlugins;
exports.createRootWithSettings = createRootWithSettings;
exports.createTestServers = createTestServers;
exports.getSupertest = getSupertest;
exports.request = void 0;
var _path = require("path");
var _loadJsonFile = _interopRequireDefault(require("load-json-file"));
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _supertest = _interopRequireDefault(require("supertest"));
var _saferLodashSet = require("@kbn/safer-lodash-set");
var _repoPackages = require("@kbn/repo-packages");
var _toolingLog = require("@kbn/tooling-log");
var _repoInfo = require("@kbn/repo-info");
var _crypto = require("crypto");
var _test = require("@kbn/test");
var _config = require("@kbn/config");
var _coreRootServerInternal = require("@kbn/core-root-server-internal");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const DEFAULTS_SETTINGS = {
  server: {
    autoListen: true,
    // Use the ephemeral port to make sure that tests use the first available
    // port and aren't affected by the timing issues in test environment.
    port: 0,
    xsrf: {
      disableProtection: true
    }
  },
  logging: {
    root: {
      level: 'off'
    }
  },
  plugins: {},
  migrations: {
    skip: false
  }
};
function createRootWithSettings(settings, cliArgs = {}, customKibanaVersion) {
  let pkg;
  if (customKibanaVersion) {
    pkg = _loadJsonFile.default.sync((0, _path.join)(_repoInfo.REPO_ROOT, 'package.json'));
    pkg.version = customKibanaVersion;
  }

  /*
   * Most of these integration tests expect OSS to default to true, but FIPS
   * requires the security plugin to be enabled
   */
  let oss = true;
  if ((0, _crypto.getFips)() === 1) {
    (0, _saferLodashSet.set)(settings, 'xpack.security.fipsMode.enabled', true);
    oss = false;
    delete cliArgs.oss;
  }
  const env = _config.Env.createDefault(_repoInfo.REPO_ROOT, {
    configs: [],
    cliArgs: {
      dev: false,
      watch: false,
      basePath: false,
      runExamples: false,
      disableOptimizer: true,
      cache: true,
      dist: false,
      oss,
      ...cliArgs
    },
    repoPackages: (0, _repoPackages.getPackages)(_repoInfo.REPO_ROOT)
  }, pkg);
  return new _coreRootServerInternal.Root({
    getConfig$: () => new _rxjs.BehaviorSubject((0, _lodash.defaultsDeep)({}, settings, DEFAULTS_SETTINGS))
  }, env);
}

/**
 * Returns supertest request attached to the core's internal native Node server.
 * @param root
 * @param method
 * @param path
 */
function getSupertest(root, method, path) {
  const testUserCredentials = Buffer.from(`${_test.systemIndicesSuperuser.username}:${_test.systemIndicesSuperuser.password}`);
  return (0, _supertest.default)(root.server.http.httpServer.server.listener)[method](path).set('Authorization', `Basic ${testUserCredentials.toString('base64')}`);
}

/**
 * Creates an instance of Root with default configuration
 * tailored for unit tests.
 *
 * @param {Object} [settings={}] Any config overrides for this instance.
 * @returns {Root}
 */
function createRoot(settings = {}, cliArgs = {}) {
  return createRootWithSettings(settings, cliArgs);
}

/**
 *  Creates an instance of Root, including all of the core plugins,
 *  with default configuration tailored for unit tests.
 *
 *  @param {Object} [settings={}] Any config overrides for this instance.
 *  @returns {Root}
 */
function createRootWithCorePlugins(settings = {}, cliArgs = {}, customKibanaVersion) {
  const DEFAULT_SETTINGS_WITH_CORE_PLUGINS = {
    elasticsearch: {
      hosts: [_test.esTestConfig.getUrl()],
      username: _test.kibanaServerTestUser.username,
      password: _test.kibanaServerTestUser.password
    },
    // Log ES deprecations to surface these in CI
    logging: {
      loggers: [{
        name: 'root',
        level: 'error',
        appenders: ['console']
      }, {
        name: 'elasticsearch.deprecation',
        level: 'all',
        appenders: ['deprecation']
      }],
      appenders: {
        deprecation: {
          type: 'console',
          layout: {
            type: 'json'
          }
        },
        console: {
          type: 'console',
          layout: {
            type: 'pattern'
          }
        }
      }
    },
    // createRootWithSettings sets default value to "true", so undefined should be threatened as "true".
    ...(cliArgs.oss === false ? {
      // reporting loads headless browser, that prevents nodejs process from exiting and causes test flakiness
      xpack: {
        reporting: {
          enabled: false
        }
      }
    } : {})
  };
  return createRootWithSettings((0, _lodash.defaultsDeep)({}, settings, DEFAULT_SETTINGS_WITH_CORE_PLUGINS), cliArgs, customKibanaVersion);
}
const request = exports.request = {
  delete: (root, path) => getSupertest(root, 'delete', path),
  get: (root, path) => getSupertest(root, 'get', path),
  head: (root, path) => getSupertest(root, 'head', path),
  post: (root, path) => getSupertest(root, 'post', path),
  put: (root, path) => getSupertest(root, 'put', path),
  patch: (root, path) => getSupertest(root, 'patch', path)
};
/**
 * Creates an instance of the Root, including all of the core "legacy" plugins,
 * with default configuration tailored for unit tests, and starts es.
 *
 * @param options
 * @prop settings Any config overrides for this instance.
 * @prop adjustTimeout A function(t) => this.timeout(t) that adjust the timeout of a
 * test, ensuring the test properly waits for the server to boot without timing out.
 */
function createTestServers({
  adjustTimeout,
  settings = {}
}) {
  var _settings$es$license, _settings$es, _settings$users, _settings$es2, _settings$kbn;
  if (!adjustTimeout) {
    throw new Error('adjustTimeout is required in order to avoid flaky tests');
  }
  let license = (_settings$es$license = (_settings$es = settings.es) === null || _settings$es === void 0 ? void 0 : _settings$es.license) !== null && _settings$es$license !== void 0 ? _settings$es$license : 'basic';
  if ((0, _crypto.getFips)() === 1) {
    // Set license to 'trial' if Node is running in FIPS mode
    license = 'trial';
  }
  const usersToBeAdded = (_settings$users = settings.users) !== null && _settings$users !== void 0 ? _settings$users : [];
  if (usersToBeAdded.length > 0) {
    if (license !== 'trial') {
      throw new Error('Adding users is only supported by createTestServers when using a trial license');
    }
  }
  const log = new _toolingLog.ToolingLog({
    level: 'debug',
    writeTo: process.stdout
  });
  const es = (0, _test.createTestEsCluster)((0, _lodash.defaultsDeep)({}, (_settings$es2 = settings.es) !== null && _settings$es2 !== void 0 ? _settings$es2 : {}, {
    log,
    license
  }));

  // Add time for KBN and adding users
  adjustTimeout(es.getStartTimeout() + 100000);
  const {
    cliArgs = {},
    customKibanaVersion,
    ...kbnSettings
  } = (_settings$kbn = settings.kbn) !== null && _settings$kbn !== void 0 ? _settings$kbn : {};
  return {
    startES: async () => {
      await es.start();
      if (['gold', 'trial'].includes(license)) {
        // Override provided configs
        kbnSettings.elasticsearch = {
          hosts: es.getHostUrls(),
          username: _test.kibanaServerTestUser.username,
          password: _test.kibanaServerTestUser.password,
          ...((0, _crypto.getFips)() ? kbnSettings.elasticsearch : {})
        };
      }
      return {
        stop: async () => await es.cleanup(),
        es,
        hosts: es.getHostUrls(),
        username: _test.kibanaServerTestUser.username,
        password: _test.kibanaServerTestUser.password
      };
    },
    startKibana: async abortSignal => {
      const root = createRootWithCorePlugins(kbnSettings, cliArgs, customKibanaVersion);
      abortSignal === null || abortSignal === void 0 ? void 0 : abortSignal.addEventListener('abort', async () => await root.shutdown());
      await root.preboot();
      const coreSetup = await root.setup();
      const coreStart = await root.start();
      return {
        root,
        coreSetup,
        coreStart,
        stop: async () => await root.shutdown()
      };
    }
  };
}