"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.performBulkGet = void 0;
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _coreElasticsearchServerInternal = require("@kbn/core-elasticsearch-server-internal");
var _coreSavedObjectsServer = require("@kbn/core-saved-objects-server");
var _coreSavedObjectsUtilsServer = require("@kbn/core-saved-objects-utils-server");
var _utils = require("../utils");
var _utils2 = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const performBulkGet = async ({
  objects,
  options
}, {
  helpers,
  allowedTypes,
  client,
  serializer,
  registry,
  extensions = {}
}) => {
  const {
    common: commonHelper,
    validation: validationHelper,
    migration: migrationHelper
  } = helpers;
  const {
    securityExtension,
    spacesExtension
  } = extensions;
  const namespace = commonHelper.getCurrentNamespace(options.namespace);
  const {
    migrationVersionCompatibility
  } = options;
  if (objects.length === 0) {
    return {
      saved_objects: []
    };
  }
  let availableSpacesPromise;
  const getAvailableSpaces = async () => {
    if (!availableSpacesPromise) {
      availableSpacesPromise = spacesExtension.getSearchableNamespaces([_coreSavedObjectsUtilsServer.ALL_NAMESPACES_STRING]).catch(err => {
        if (_boom.default.isBoom(err) && err.output.payload.statusCode === 403) {
          // the user doesn't have access to any spaces; return the current space ID and allow the SOR authZ check to fail
          return [_coreSavedObjectsUtilsServer.SavedObjectsUtils.namespaceIdToString(namespace)];
        } else {
          throw err;
        }
      });
    }
    return availableSpacesPromise;
  };
  let bulkGetRequestIndexCounter = 0;
  const expectedBulkGetResults = await Promise.all(objects.map(async object => {
    var _namespaces;
    const {
      type,
      id,
      fields
    } = object;
    let error;
    if (!allowedTypes.includes(type)) {
      error = _coreSavedObjectsServer.SavedObjectsErrorHelpers.createUnsupportedTypeError(type);
    } else {
      try {
        validationHelper.validateObjectNamespaces(type, id, object.namespaces);
      } catch (e) {
        error = e;
      }
    }
    if (error) {
      return (0, _utils2.left)({
        id,
        type,
        error: (0, _utils2.errorContent)(error)
      });
    }
    let namespaces = object.namespaces;
    if (spacesExtension && (_namespaces = namespaces) !== null && _namespaces !== void 0 && _namespaces.includes(_coreSavedObjectsUtilsServer.ALL_NAMESPACES_STRING)) {
      namespaces = await getAvailableSpaces();
    }
    const getFields = savedObjectFields => {
      const isEmpty = !savedObjectFields || savedObjectFields.length === 0;
      if (securityExtension && securityExtension.includeSavedObjectNames() && !isEmpty) {
        const nameAttribute = registry.getNameAttribute(type);
        const nameFields = nameAttribute !== 'unknown' ? [nameAttribute] : ['name', 'title'];
        return [...savedObjectFields, ...nameFields];
      }
      return fields;
    };
    return (0, _utils2.right)({
      type,
      id,
      fields: getFields(fields),
      namespaces,
      esRequestIndex: bulkGetRequestIndexCounter++
    });
  }));
  const validObjects = expectedBulkGetResults.filter(_utils2.isRight);
  if (validObjects.length === 0) {
    // We only have error results; return early to avoid potentially trying authZ checks for 0 types which would result in an exception.
    return {
      // Technically the returned array should only contain SavedObject results, but for errors this is not true (we cast to 'any' below)
      saved_objects: expectedBulkGetResults.map(({
        value
      }) => value)
    };
  }
  const getNamespaceId = namespaces => namespaces !== undefined ? _coreSavedObjectsUtilsServer.SavedObjectsUtils.namespaceStringToId(namespaces[0]) : namespace;
  const bulkGetDocs = validObjects.map(({
    value: {
      type,
      id,
      fields,
      namespaces
    }
  }) => ({
    _id: serializer.generateRawId(getNamespaceId(namespaces), type, id),
    // the namespace prefix is only used for single-namespace object types
    _index: commonHelper.getIndexForType(type),
    _source: {
      includes: (0, _utils.includedFields)(type, fields)
    }
  }));
  const bulkGetResponse = bulkGetDocs.length ? await client.mget({
    docs: bulkGetDocs
  }, {
    ignore: [404],
    meta: true
  }) : undefined;
  // fail fast if we can't verify a 404 is from Elasticsearch
  if (bulkGetResponse && (0, _coreElasticsearchServerInternal.isNotFoundFromUnsupportedServer)({
    statusCode: bulkGetResponse.statusCode,
    headers: bulkGetResponse.headers
  })) {
    throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundEsUnavailableError();
  }
  const authObjects = [];
  const documents = expectedBulkGetResults.map(expectedResult => {
    var _doc$_source$namespac, _doc$_source;
    if ((0, _utils2.isLeft)(expectedResult)) {
      const {
        type,
        id
      } = expectedResult.value;
      authObjects.push({
        type,
        id,
        existingNamespaces: [],
        error: true
      });
      return expectedResult.value;
    }
    const {
      type,
      id,
      // set to default namespaces value for `rawDocExistsInNamespaces` check below
      namespaces = [_coreSavedObjectsUtilsServer.SavedObjectsUtils.namespaceIdToString(namespace)],
      esRequestIndex
    } = expectedResult.value;
    const doc = bulkGetResponse === null || bulkGetResponse === void 0 ? void 0 : bulkGetResponse.body.docs[esRequestIndex];

    // @ts-expect-error MultiGetHit._source is optional
    const docNotFound = !(doc !== null && doc !== void 0 && doc.found) || !(0, _utils2.rawDocExistsInNamespaces)(registry, doc, namespaces);
    const savedObject = docNotFound ? {
      id,
      type,
      error: (0, _utils2.errorContent)(_coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id))
    } :
    // @ts-expect-error MultiGetHit._source is optional
    (0, _utils2.getSavedObjectFromSource)(registry, type, id, doc, {
      migrationVersionCompatibility
    });
    authObjects.push({
      type,
      id,
      objectNamespaces: namespaces,
      // @ts-expect-error MultiGetHit._source is optional
      existingNamespaces: (_doc$_source$namespac = doc === null || doc === void 0 ? void 0 : (_doc$_source = doc._source) === null || _doc$_source === void 0 ? void 0 : _doc$_source.namespaces) !== null && _doc$_source$namespac !== void 0 ? _doc$_source$namespac : [],
      error: docNotFound,
      name: !docNotFound ? _coreSavedObjectsUtilsServer.SavedObjectsUtils.getName(registry.getNameAttribute(type), savedObject) : undefined
    });
    return savedObject;
  });
  const authorizationResult = await (securityExtension === null || securityExtension === void 0 ? void 0 : securityExtension.authorizeBulkGet({
    namespace,
    objects: authObjects
  }));
  const results = [];
  for (const doc of documents) {
    results.push(doc.error ? doc : await migrationHelper.migrateAndDecryptStorageDocument({
      document: doc,
      typeMap: authorizationResult === null || authorizationResult === void 0 ? void 0 : authorizationResult.typeMap
    }));
  }
  return {
    saved_objects: results
  };
};
exports.performBulkGet = performBulkGet;