"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.adoptToHapiAuthFormat = adoptToHapiAuthFormat;
var _coreHttpServer = require("@kbn/core-http-server");
var _coreHttpRouterServerInternal = require("@kbn/core-http-router-server-internal");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const authResult = {
  authenticated(data = {}) {
    return {
      type: _coreHttpServer.AuthResultType.authenticated,
      state: data.state,
      requestHeaders: data.requestHeaders,
      responseHeaders: data.responseHeaders
    };
  },
  notHandled() {
    return {
      type: _coreHttpServer.AuthResultType.notHandled
    };
  },
  redirected(headers) {
    return {
      type: _coreHttpServer.AuthResultType.redirected,
      headers
    };
  },
  isAuthenticated(result) {
    return (result === null || result === void 0 ? void 0 : result.type) === _coreHttpServer.AuthResultType.authenticated;
  },
  isNotHandled(result) {
    return (result === null || result === void 0 ? void 0 : result.type) === _coreHttpServer.AuthResultType.notHandled;
  },
  isRedirected(result) {
    return (result === null || result === void 0 ? void 0 : result.type) === _coreHttpServer.AuthResultType.redirected;
  }
};
const toolkit = {
  authenticated: authResult.authenticated,
  notHandled: authResult.notHandled,
  redirected: authResult.redirected
};

/** @internal */
function adoptToHapiAuthFormat(fn, log, onAuth = () => undefined) {
  return async function interceptAuth(request, responseToolkit) {
    const hapiResponseAdapter = new _coreHttpRouterServerInternal.HapiResponseAdapter(responseToolkit);
    const kibanaRequest = _coreHttpRouterServerInternal.CoreKibanaRequest.from(request, undefined, false);
    try {
      const result = await fn(kibanaRequest, _coreHttpRouterServerInternal.lifecycleResponseFactory, toolkit);
      if ((0, _coreHttpServer.isKibanaResponse)(result)) {
        return hapiResponseAdapter.handle(result);
      }
      if (authResult.isAuthenticated(result)) {
        onAuth(request, {
          state: result.state,
          requestHeaders: result.requestHeaders,
          responseHeaders: result.responseHeaders
        });
        return responseToolkit.authenticated({
          credentials: result.state || {}
        });
      }
      if (authResult.isRedirected(result)) {
        // we cannot redirect a user when resources with optional auth requested
        if (kibanaRequest.route.options.authRequired === 'optional') {
          return responseToolkit.continue;
        }
        return hapiResponseAdapter.handle(_coreHttpRouterServerInternal.lifecycleResponseFactory.redirected({
          // hapi doesn't accept string[] as a valid header
          headers: result.headers
        }));
      }
      if (authResult.isNotHandled(result)) {
        if (kibanaRequest.route.options.authRequired === 'optional') {
          return responseToolkit.continue;
        }
        return hapiResponseAdapter.handle(_coreHttpRouterServerInternal.lifecycleResponseFactory.unauthorized());
      }
      throw new Error(`Unexpected result from Authenticate. Expected AuthResult or KibanaResponse, but given: ${result}.`);
    } catch (error) {
      log.error(error);
      return hapiResponseAdapter.toInternalError();
    }
  };
}