"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PerAlertActionScheduler = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _alertingTypes = require("@kbn/alerting-types");
var _lodash = require("lodash");
var _common = require("../../../../common");
var _lib = require("../lib");
var _transform_action_params = require("../../transform_action_params");
var _inject_action_params = require("../../inject_action_params");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
var Reasons = /*#__PURE__*/function (Reasons) {
  Reasons["MUTED"] = "muted";
  Reasons["THROTTLED"] = "throttled";
  Reasons["ACTION_GROUP_NOT_CHANGED"] = "actionGroupHasNotChanged";
  return Reasons;
}(Reasons || {});
class PerAlertActionScheduler {
  constructor(context) {
    var _context$rule$actions;
    (0, _defineProperty2.default)(this, "actions", []);
    (0, _defineProperty2.default)(this, "mutedAlertIdsSet", new Set());
    (0, _defineProperty2.default)(this, "ruleTypeActionGroups", void 0);
    (0, _defineProperty2.default)(this, "skippedAlerts", {});
    this.context = context;
    this.ruleTypeActionGroups = new Map(context.ruleType.actionGroups.map(actionGroup => [actionGroup.id, actionGroup.name]));
    this.mutedAlertIdsSet = new Set(context.rule.mutedInstanceIds);
    const canGetSummarizedAlerts = !!context.ruleType.alerts && !!context.alertsClient.getSummarizedAlerts;

    // filter for per-alert actions; if the action has an alertsFilter, check that
    // rule type supports summarized alerts and filter out if not
    this.actions = (0, _lodash.compact)(((_context$rule$actions = context.rule.actions) !== null && _context$rule$actions !== void 0 ? _context$rule$actions : []).filter(action => !(0, _lib.isSummaryAction)(action)).map(action => {
      if (!canGetSummarizedAlerts && action.alertsFilter) {
        this.context.logger.error(`Skipping action "${action.id}" for rule "${this.context.rule.id}" because the rule type "${this.context.ruleType.name}" does not support alert-as-data.`);
        return null;
      }
      return action;
    }));
  }
  get priority() {
    return 2;
  }
  async getActionsToSchedule({
    activeAlerts,
    recoveredAlerts
  }) {
    const executables = [];
    const results = [];
    const activeAlertsArray = Object.values(activeAlerts || {});
    const recoveredAlertsArray = Object.values(recoveredAlerts || {});
    for (const action of this.actions) {
      let summarizedAlerts = null;
      if (action.useAlertDataForTemplate || action.alertsFilter) {
        const optionsBase = {
          spaceId: this.context.taskInstance.params.spaceId,
          ruleId: this.context.rule.id,
          excludedAlertInstanceIds: this.context.rule.mutedInstanceIds,
          alertsFilter: action.alertsFilter
        };
        let options;
        if ((0, _lib.isActionOnInterval)(action)) {
          const throttleMills = (0, _common.parseDuration)(action.frequency.throttle);
          const start = new Date(Date.now() - throttleMills);
          options = {
            ...optionsBase,
            start,
            end: new Date()
          };
        } else {
          options = {
            ...optionsBase,
            executionUuid: this.context.executionId
          };
        }
        summarizedAlerts = await (0, _lib.getSummarizedAlerts)({
          queryOptions: options,
          alertsClient: this.context.alertsClient
        });
        (0, _lib.logNumberOfFilteredAlerts)({
          logger: this.context.logger,
          numberOfAlerts: activeAlertsArray.length + recoveredAlertsArray.length,
          numberOfSummarizedAlerts: summarizedAlerts.all.count,
          action
        });
      }
      for (const alert of activeAlertsArray) {
        if (this.isExecutableAlert({
          alert,
          action,
          summarizedAlerts
        }) && this.isExecutableActiveAlert({
          alert,
          action
        })) {
          this.addSummarizedAlerts({
            alert,
            summarizedAlerts
          });
          executables.push({
            action,
            alert
          });
        }
      }
      if (this.isRecoveredAction(action.group)) {
        for (const alert of recoveredAlertsArray) {
          if (this.isExecutableAlert({
            alert,
            action,
            summarizedAlerts
          })) {
            this.addSummarizedAlerts({
              alert,
              summarizedAlerts
            });
            executables.push({
              action,
              alert
            });
          }
        }
      }
    }
    if (executables.length === 0) return [];
    this.context.ruleRunMetricsStore.incrementNumberOfGeneratedActions(executables.length);
    const ruleUrl = (0, _lib.buildRuleUrl)({
      getViewInAppRelativeUrl: this.context.ruleType.getViewInAppRelativeUrl,
      kibanaBaseUrl: this.context.taskRunnerContext.kibanaBaseUrl,
      logger: this.context.logger,
      rule: this.context.rule,
      spaceId: this.context.taskInstance.params.spaceId
    });
    for (const {
      action,
      alert
    } of executables) {
      const {
        actionTypeId
      } = action;
      if (!(0, _lib.shouldScheduleAction)({
        action,
        actionsConfigMap: this.context.taskRunnerContext.actionsConfigMap,
        isActionExecutable: this.context.taskRunnerContext.actionsPlugin.isActionExecutable,
        logger: this.context.logger,
        ruleId: this.context.rule.id,
        ruleRunMetricsStore: this.context.ruleRunMetricsStore
      })) {
        continue;
      }
      this.context.ruleRunMetricsStore.incrementNumberOfTriggeredActions();
      this.context.ruleRunMetricsStore.incrementNumberOfTriggeredActionsByConnectorType(actionTypeId);
      const actionGroup = action.group;
      const transformActionParamsOptions = {
        actionsPlugin: this.context.taskRunnerContext.actionsPlugin,
        alertId: this.context.rule.id,
        alertType: this.context.ruleType.id,
        actionTypeId: action.actionTypeId,
        alertName: this.context.rule.name,
        spaceId: this.context.taskInstance.params.spaceId,
        tags: this.context.rule.tags,
        alertInstanceId: alert.getId(),
        alertUuid: alert.getUuid(),
        alertActionGroup: actionGroup,
        alertActionGroupName: this.ruleTypeActionGroups.get(actionGroup),
        context: alert.getContext(),
        actionId: action.id,
        state: alert.getState(),
        kibanaBaseUrl: this.context.taskRunnerContext.kibanaBaseUrl,
        alertParams: this.context.rule.params,
        actionParams: action.params,
        flapping: alert.getFlapping(),
        ruleUrl: ruleUrl === null || ruleUrl === void 0 ? void 0 : ruleUrl.absoluteUrl,
        consecutiveMatches: alert.getActiveCount()
      };
      if (alert.isAlertAsData()) {
        transformActionParamsOptions.aadAlert = alert.getAlertAsData();
      }
      const actionToRun = {
        ...action,
        params: (0, _inject_action_params.injectActionParams)({
          actionTypeId: action.actionTypeId,
          ruleUrl,
          ruleName: this.context.rule.name,
          actionParams: (0, _transform_action_params.transformActionParams)(transformActionParamsOptions)
        })
      };
      results.push({
        actionToEnqueue: (0, _lib.formatActionToEnqueue)({
          action: actionToRun,
          apiKey: this.context.apiKey,
          apiKeyId: this.context.apiKeyId,
          executionId: this.context.executionId,
          priority: this.context.priority,
          ruleConsumer: this.context.ruleConsumer,
          ruleId: this.context.rule.id,
          ruleTypeId: this.context.ruleType.id,
          spaceId: this.context.taskInstance.params.spaceId
        }),
        actionToLog: {
          id: action.id,
          // uuid is typed as optional but in reality it is always
          // populated - https://github.com/elastic/kibana/issues/195255
          uuid: action.uuid,
          typeId: action.actionTypeId,
          alertId: alert.getId(),
          alertGroup: action.group
        }
      });
      if (!this.isRecoveredAction(actionGroup)) {
        if ((0, _lib.isActionOnInterval)(action)) {
          alert.updateLastScheduledActions(action.group, (0, _lib.generateActionHash)(action), action.uuid);
        } else {
          alert.updateLastScheduledActions(action.group);
        }
        alert.unscheduleActions();
      }
    }
    return results;
  }
  isExecutableAlert({
    alert,
    action,
    summarizedAlerts
  }) {
    return !this.hasActiveMaintenanceWindow({
      alert,
      action
    }) && !this.isAlertMuted(alert) && !this.hasPendingCountButNotNotifyOnChange({
      alert,
      action
    }) && !alert.isFilteredOut(summarizedAlerts);
  }
  isExecutableActiveAlert({
    alert,
    action
  }) {
    var _alert$getScheduledAc, _action$frequency;
    if (!alert.hasScheduledActions()) {
      return false;
    }
    const alertsActionGroup = (_alert$getScheduledAc = alert.getScheduledActionOptions()) === null || _alert$getScheduledAc === void 0 ? void 0 : _alert$getScheduledAc.actionGroup;
    if (!this.isValidActionGroup(alertsActionGroup)) {
      return false;
    }
    if (action.group !== alertsActionGroup) {
      return false;
    }
    const alertId = alert.getId();
    const {
      context: {
        rule,
        logger,
        ruleLabel
      }
    } = this;
    const notifyWhen = ((_action$frequency = action.frequency) === null || _action$frequency === void 0 ? void 0 : _action$frequency.notifyWhen) || rule.notifyWhen;
    if (notifyWhen === 'onActionGroupChange' && !alert.scheduledActionGroupHasChanged()) {
      if (!this.skippedAlerts[alertId] || this.skippedAlerts[alertId] && this.skippedAlerts[alertId].reason !== Reasons.ACTION_GROUP_NOT_CHANGED) {
        logger.debug(`skipping scheduling of actions for '${alertId}' in rule ${ruleLabel}: alert is active but action group has not changed`);
      }
      this.skippedAlerts[alertId] = {
        reason: Reasons.ACTION_GROUP_NOT_CHANGED
      };
      return false;
    }
    if (notifyWhen === 'onThrottleInterval') {
      var _action$frequency2, _action$frequency$thr, _rule$throttle;
      const throttled = (_action$frequency2 = action.frequency) !== null && _action$frequency2 !== void 0 && _action$frequency2.throttle ? alert.isThrottled({
        throttle: (_action$frequency$thr = action.frequency.throttle) !== null && _action$frequency$thr !== void 0 ? _action$frequency$thr : null,
        actionHash: (0, _lib.generateActionHash)(action),
        // generateActionHash must be removed once all the hash identifiers removed from the task state
        uuid: action.uuid
      }) : alert.isThrottled({
        throttle: (_rule$throttle = rule.throttle) !== null && _rule$throttle !== void 0 ? _rule$throttle : null
      });
      if (throttled) {
        if (!this.skippedAlerts[alertId] || this.skippedAlerts[alertId] && this.skippedAlerts[alertId].reason !== Reasons.THROTTLED) {
          logger.debug(`skipping scheduling of actions for '${alertId}' in rule ${ruleLabel}: rule is throttled`);
        }
        this.skippedAlerts[alertId] = {
          reason: Reasons.THROTTLED
        };
        return false;
      }
    }
    return true;
  }
  isRecoveredAction(actionGroup) {
    return actionGroup === this.context.ruleType.recoveryActionGroup.id;
  }
  isAlertMuted(alert) {
    const alertId = alert.getId();
    const muted = this.mutedAlertIdsSet.has(alertId);
    if (muted) {
      if (!this.skippedAlerts[alertId] || this.skippedAlerts[alertId] && this.skippedAlerts[alertId].reason !== Reasons.MUTED) {
        this.context.logger.debug(`skipping scheduling of actions for '${alertId}' in rule ${this.context.ruleLabel}: rule is muted`);
      }
      this.skippedAlerts[alertId] = {
        reason: Reasons.MUTED
      };
      return true;
    }
    return false;
  }
  isValidActionGroup(actionGroup) {
    if (!this.ruleTypeActionGroups.has(actionGroup)) {
      this.context.logger.error(`Invalid action group "${actionGroup}" for rule "${this.context.ruleType.id}".`);
      return false;
    }
    return true;
  }
  hasActiveMaintenanceWindow({
    alert,
    action
  }) {
    const alertMaintenanceWindowIds = alert.getMaintenanceWindowIds();
    if (alertMaintenanceWindowIds.length !== 0) {
      this.context.logger.debug(`no scheduling of actions "${action.id}" for alert "${alert.getId()}" from rule "${this.context.rule.id}": has active maintenance windows ${alertMaintenanceWindowIds.join(', ')}.`);
      return true;
    }
    return false;
  }
  addSummarizedAlerts({
    alert,
    summarizedAlerts
  }) {
    if (summarizedAlerts) {
      const alertAsData = summarizedAlerts.all.data.find(alertHit => alertHit._id === alert.getUuid());
      if (alertAsData) {
        alert.setAlertAsData(alertAsData);
      }
    }
  }
  hasPendingCountButNotNotifyOnChange({
    alert,
    action
  }) {
    var _action$frequency3;
    // only actions with notifyWhen set to "on status change" should return
    // notifications for flapping pending recovered alerts
    if (alert.getPendingRecoveredCount() > 0 && (action === null || action === void 0 ? void 0 : (_action$frequency3 = action.frequency) === null || _action$frequency3 === void 0 ? void 0 : _action$frequency3.notifyWhen) !== _alertingTypes.RuleNotifyWhen.CHANGE) {
      return true;
    }
    return false;
  }
}
exports.PerAlertActionScheduler = PerAlertActionScheduler;