"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.enableSpaceAwarenessMigration = enableSpaceAwarenessMigration;
var _server = require("@kbn/core/server");
var _common = require("../../../common");
var _constants = require("../../../common/constants");
var _ = require("..");
var _constants2 = require("../../constants");
var _settings = require("../settings");
var _errors = require("../../errors");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function enableSpaceAwarenessMigration() {
  const soClient = _.appContextService.getInternalUserSOClientWithoutSpaceExtension();
  const logger = _.appContextService.getLogger();
  const existingSettings = await (0, _settings.getSettingsOrUndefined)(soClient);
  if ((existingSettings === null || existingSettings === void 0 ? void 0 : existingSettings.use_space_awareness_migration_status) === 'success') {
    return;
  }
  if (existingSettings !== null && existingSettings !== void 0 && existingSettings.use_space_awareness_migration_started_at && new Date(existingSettings === null || existingSettings === void 0 ? void 0 : existingSettings.use_space_awareness_migration_started_at).getTime() > Date.now() - _helpers.PENDING_MIGRATION_TIMEOUT) {
    logger.info('Fleet space awareness migration is pending');
    throw new _errors.FleetError('Migration is pending.');
  }
  await (0, _settings.saveSettings)(soClient, {
    use_space_awareness_migration_status: 'pending',
    use_space_awareness_migration_started_at: new Date().toISOString()
  }, {
    createWithOverwrite: false,
    version: existingSettings === null || existingSettings === void 0 ? void 0 : existingSettings.version
  }).catch(err => {
    if (_server.SavedObjectsErrorHelpers.isConflictError(err)) {
      logger.info('Fleet space awareness migration is pending');
      throw new _errors.FleetError('Migration is pending. (conflict acquiring the lock)');
    }
    throw err;
  });
  await runMigration(soClient, logger).then(async () => {
    logger.info('Fleet space awareness migration is complete');
    // Update Settings SO
    await (0, _settings.saveSettings)(soClient, {
      use_space_awareness_migration_status: 'success',
      use_space_awareness_migration_started_at: null
    });
  }).catch(async error => {
    logger.error('Fleet space awareness migration failed', {
      error
    });
    await (0, _settings.saveSettings)(soClient, {
      use_space_awareness_migration_status: 'error',
      use_space_awareness_migration_started_at: null
    });
    throw error;
  });
}
async function runMigration(soClient, logger) {
  logger.info('Starting Fleet space awareness migration');
  // Agent Policy
  await batchMigration(soClient, _common.LEGACY_AGENT_POLICY_SAVED_OBJECT_TYPE, _constants2.AGENT_POLICY_SAVED_OBJECT_TYPE);
  // Package policu
  await batchMigration(soClient, _constants.LEGACY_PACKAGE_POLICY_SAVED_OBJECT_TYPE, _constants.PACKAGE_POLICY_SAVED_OBJECT_TYPE);
}
const BATCH_SIZE = 1000;
async function batchMigration(soClient, sourceSoType, targetSoType) {
  const finder = soClient.createPointInTimeFinder({
    type: sourceSoType,
    perPage: BATCH_SIZE
  });
  try {
    for await (const result of finder.find()) {
      const createRes = await soClient.bulkCreate(result.saved_objects.map(so => ({
        type: targetSoType,
        id: so.id,
        attributes: so.attributes
      })), {
        overwrite: true,
        refresh: 'wait_for'
      });
      for (const res of createRes.saved_objects) {
        if (res.error) {
          throw res.error;
        }
      }
    }
  } finally {
    await finder.close();
  }
}