"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.generateConnectorName = void 0;
var _uuid = require("uuid");
var _error_codes = require("../../../common/types/error_codes");
var _to_alphanumeric = require("../../../common/utils/to_alphanumeric");
var _exists_index = require("../indices/exists_index");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const generateConnectorName = async (client, connectorType, userConnectorName) => {
  const prefix = (0, _to_alphanumeric.toAlphanumeric)(connectorType);
  if (!prefix || prefix.length === 0) {
    throw new Error('Connector type or connectorName is required');
  }
  if (userConnectorName) {
    let indexName = `connector-${userConnectorName}`;
    const resultSameName = await (0, _exists_index.indexOrAliasExists)(client, indexName);
    // index with same name doesn't exist
    if (!resultSameName) {
      return {
        apiKeyName: userConnectorName,
        connectorName: userConnectorName,
        indexName
      };
    }
    // if the index name already exists, we will generate until it doesn't for 20 times
    for (let i = 0; i < 20; i++) {
      indexName = `connector-${userConnectorName}-${(0, _uuid.v4)().split('-')[1].slice(0, 4)}`;
      const result = await (0, _exists_index.indexOrAliasExists)(client, indexName);
      if (!result) {
        return {
          apiKeyName: indexName,
          connectorName: userConnectorName,
          indexName
        };
      }
    }
  } else {
    for (let i = 0; i < 20; i++) {
      const connectorName = `${prefix}-${(0, _uuid.v4)().split('-')[1].slice(0, 4)}`;
      const indexName = `connector-${connectorName}`;
      const result = await (0, _exists_index.indexOrAliasExists)(client, indexName);
      if (!result) {
        return {
          apiKeyName: indexName,
          connectorName,
          indexName
        };
      }
    }
  }
  throw new Error(_error_codes.ErrorCode.GENERATE_INDEX_NAME_ERROR);
};
exports.generateConnectorName = generateConnectorName;