"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.bulkActionKnowledgeBaseEntriesRoute = void 0;
var _moment = _interopRequireDefault(require("moment"));
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _common = require("@kbn/elastic-assistant-common/impl/schemas/common");
var _audit_events = require("../../../ai_assistant_data_clients/knowledge_base/audit_events");
var _event_based_telemetry = require("../../../lib/telemetry/event_based_telemetry");
var _helpers = require("../../helpers");
var _constants = require("../../../../common/constants");
var _utils = require("../../utils");
var _transforms = require("../../../ai_assistant_data_clients/knowledge_base/transforms");
var _create_knowledge_base_entry = require("../../../ai_assistant_data_clients/knowledge_base/create_knowledge_base_entry");
var _utils2 = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const buildBulkResponse = (response, {
  errors = [],
  updated = [],
  created = [],
  deleted = [],
  skipped = []
}, telemetry, auditLogger) => {
  const numSucceeded = updated.length + created.length + deleted.length;
  const numSkipped = skipped.length;
  const numFailed = errors.length;
  const summary = {
    failed: numFailed,
    succeeded: numSucceeded,
    skipped: numSkipped,
    total: numSucceeded + numFailed + numSkipped
  };
  const results = {
    updated,
    created,
    deleted,
    skipped
  };
  if (created.length) {
    created.forEach(entry => {
      var _entry$required;
      telemetry.reportEvent(_event_based_telemetry.CREATE_KNOWLEDGE_BASE_ENTRY_SUCCESS_EVENT.eventType, {
        entryType: entry.type,
        required: 'required' in entry ? (_entry$required = entry.required) !== null && _entry$required !== void 0 ? _entry$required : false : false,
        sharing: entry.users.length ? 'private' : 'global',
        ...(entry.type === 'document' ? {
          source: entry.source
        } : {})
      });
      auditLogger === null || auditLogger === void 0 ? void 0 : auditLogger.log((0, _audit_events.knowledgeBaseAuditEvent)({
        action: _audit_events.KnowledgeBaseAuditAction.CREATE,
        id: entry.id,
        name: entry.name,
        outcome: _audit_events.AUDIT_OUTCOME.SUCCESS
      }));
    });
  }
  if (updated.length) {
    updated.forEach(entry => {
      auditLogger === null || auditLogger === void 0 ? void 0 : auditLogger.log((0, _audit_events.knowledgeBaseAuditEvent)({
        action: _audit_events.KnowledgeBaseAuditAction.UPDATE,
        id: entry.id,
        name: entry.name,
        outcome: _audit_events.AUDIT_OUTCOME.SUCCESS
      }));
    });
  }
  if (deleted.length) {
    deleted.forEach(deletedId => {
      auditLogger === null || auditLogger === void 0 ? void 0 : auditLogger.log((0, _audit_events.knowledgeBaseAuditEvent)({
        action: _audit_events.KnowledgeBaseAuditAction.DELETE,
        id: deletedId,
        outcome: _audit_events.AUDIT_OUTCOME.SUCCESS
      }));
    });
  }
  if (numFailed > 0) {
    return response.custom({
      headers: {
        'content-type': 'application/json'
      },
      body: {
        message: summary.succeeded > 0 ? 'Bulk edit partially failed' : 'Bulk edit failed',
        attributes: {
          errors: errors.map(e => {
            var _e$status;
            return {
              statusCode: (_e$status = e.status) !== null && _e$status !== void 0 ? _e$status : 500,
              knowledgeBaseEntries: [{
                id: e.document.id,
                name: ''
              }],
              message: e.message
            };
          }),
          results,
          summary
        }
      },
      statusCode: 500
    });
  }
  const responseBody = {
    success: true,
    knowledgeBaseEntriesCount: summary.total,
    attributes: {
      results,
      summary
    }
  };
  return response.ok({
    body: responseBody
  });
};
const bulkActionKnowledgeBaseEntriesRoute = router => {
  router.versioned.post({
    access: 'public',
    path: _elasticAssistantCommon.ELASTIC_AI_ASSISTANT_KNOWLEDGE_BASE_ENTRIES_URL_BULK_ACTION,
    security: {
      authz: {
        requiredPrivileges: ['elasticAssistant']
      }
    },
    options: {
      timeout: {
        idleSocket: _moment.default.duration(15, 'minutes').asMilliseconds()
      }
    }
  }).addVersion({
    version: _elasticAssistantCommon.API_VERSIONS.public.v1,
    validate: {
      request: {
        body: (0, _common.buildRouteValidationWithZod)(_elasticAssistantCommon.PerformKnowledgeBaseEntryBulkActionRequestBody)
      }
    }
  }, async (context, request, response) => {
    const assistantResponse = (0, _utils.buildResponse)(response);
    try {
      var _body$update, _body$create, _body$delete$ids$leng, _body$delete, _body$delete$ids, _body$delete$ids2, _body$delete2, _body$update$map, _body$update2, _body$create2, _body$delete3, _body$update3;
      const ctx = await context.resolve(['core', 'elasticAssistant', 'licensing']);
      const logger = ctx.elasticAssistant.logger;

      // Perform license, authenticated user and FF checks
      const checkResponse = await (0, _helpers.performChecks)({
        context: ctx,
        request,
        response
      });
      if (!checkResponse.isSuccess) {
        return checkResponse.response;
      }
      logger.debug(() => `Performing bulk action on Knowledge Base Entries:\n${JSON.stringify(request.body)}`);
      const {
        body
      } = request;
      const operationsCount = (body !== null && body !== void 0 && body.update ? (_body$update = body.update) === null || _body$update === void 0 ? void 0 : _body$update.length : 0) + (body !== null && body !== void 0 && body.create ? (_body$create = body.create) === null || _body$create === void 0 ? void 0 : _body$create.length : 0) + (body !== null && body !== void 0 && body.delete ? (_body$delete$ids$leng = (_body$delete = body.delete) === null || _body$delete === void 0 ? void 0 : (_body$delete$ids = _body$delete.ids) === null || _body$delete$ids === void 0 ? void 0 : _body$delete$ids.length) !== null && _body$delete$ids$leng !== void 0 ? _body$delete$ids$leng : 0 : 0);
      if (operationsCount > _constants.KNOWLEDGE_BASE_ENTRIES_TABLE_MAX_PAGE_SIZE) {
        return assistantResponse.error({
          body: `More than ${_constants.KNOWLEDGE_BASE_ENTRIES_TABLE_MAX_PAGE_SIZE} ids sent for bulk edit action.`,
          statusCode: 400
        });
      }
      const abortController = new AbortController();

      // subscribing to completed$, because it handles both cases when request was completed and aborted.
      // when route is finished by timeout, aborted$ is not getting fired
      request.events.completed$.subscribe(() => abortController.abort());
      const kbDataClient = await ctx.elasticAssistant.getAIAssistantKnowledgeBaseDataClient();
      const spaceId = ctx.elasticAssistant.getSpaceId();
      const authenticatedUser = checkResponse.currentUser;
      const manageGlobalKnowledgeBaseAIAssistant = kbDataClient === null || kbDataClient === void 0 ? void 0 : kbDataClient.options.manageGlobalKnowledgeBaseAIAssistant;
      if (body.create && body.create.length > 0) {
        // RBAC validation
        body.create.forEach(entry => {
          if ((0, _utils2.isGlobalEntry)(entry) && !manageGlobalKnowledgeBaseAIAssistant) {
            throw new Error(`User lacks privileges to create global knowledge base entries`);
          }
        });
      }
      if (body.update && body.update.length > 0) {
        body.update.forEach(entry => {
          if ((0, _utils2.isGlobalEntry)(entry) && !manageGlobalKnowledgeBaseAIAssistant) {
            throw new Error(`User lacks privileges to create global knowledge base entries`);
          }
        });
      }
      await (0, _utils2.validateDocumentsModification)(kbDataClient, authenticatedUser, (_body$delete$ids2 = (_body$delete2 = body.delete) === null || _body$delete2 === void 0 ? void 0 : _body$delete2.ids) !== null && _body$delete$ids2 !== void 0 ? _body$delete$ids2 : [], 'delete');
      await (0, _utils2.validateDocumentsModification)(kbDataClient, authenticatedUser, (_body$update$map = (_body$update2 = body.update) === null || _body$update2 === void 0 ? void 0 : _body$update2.map(entry => entry.id)) !== null && _body$update$map !== void 0 ? _body$update$map : [], 'update');
      const writer = await (kbDataClient === null || kbDataClient === void 0 ? void 0 : kbDataClient.getWriter());
      const changedAt = new Date().toISOString();
      const {
        errors,
        docs_created: docsCreated,
        docs_updated: docsUpdated,
        docs_deleted: docsDeleted
        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
      } = await writer.bulk({
        documentsToCreate: (_body$create2 = body.create) === null || _body$create2 === void 0 ? void 0 : _body$create2.map(entry => (0, _create_knowledge_base_entry.transformToCreateSchema)({
          createdAt: changedAt,
          spaceId,
          user: authenticatedUser,
          entry
        })),
        documentsToDelete: (_body$delete3 = body.delete) === null || _body$delete3 === void 0 ? void 0 : _body$delete3.ids,
        documentsToUpdate: (_body$update3 = body.update) === null || _body$update3 === void 0 ? void 0 : _body$update3.map(entry => (0, _create_knowledge_base_entry.transformToUpdateSchema)({
          user: authenticatedUser,
          updatedAt: changedAt,
          entry
        })),
        getUpdateScript: entry => (0, _create_knowledge_base_entry.getUpdateScript)({
          entry
        }),
        authenticatedUser
      });
      const created = docsCreated.length > 0 ? await (kbDataClient === null || kbDataClient === void 0 ? void 0 : kbDataClient.findDocuments({
        page: 1,
        perPage: 100,
        filter: docsCreated.map(c => `_id:${c}`).join(' OR ')
      })) : undefined;
      return buildBulkResponse(response, {
        // @ts-ignore-next-line TS2322
        updated: (0, _transforms.transformESToKnowledgeBase)(docsUpdated),
        created: created !== null && created !== void 0 && created.data ? (0, _transforms.transformESSearchToKnowledgeBaseEntry)(created === null || created === void 0 ? void 0 : created.data) : [],
        deleted: docsDeleted !== null && docsDeleted !== void 0 ? docsDeleted : [],
        skipped: [],
        errors
      }, ctx.elasticAssistant.telemetry, ctx.elasticAssistant.auditLogger);
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return assistantResponse.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.bulkActionKnowledgeBaseEntriesRoute = bulkActionKnowledgeBaseEntriesRoute;