"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.postAttackDiscoveryRoute = void 0;
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _common = require("@kbn/elastic-assistant-common/impl/schemas/common");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _uuid = require("uuid");
var _constants = require("../../../../common/constants");
var _get_duration_nanoseconds = require("./get_duration_nanoseconds");
var _helpers = require("../../helpers");
var _helpers2 = require("../helpers/helpers");
var _write_attack_discovery_event = require("./helpers/write_attack_discovery_event");
var _build_response = require("../../../lib/build_response");
var _request_is_valid = require("./helpers/request_is_valid");
var _generate_and_update_discoveries = require("../helpers/generate_and_update_discoveries");
var _index_privileges = require("../helpers/index_privileges");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ROUTE_HANDLER_TIMEOUT = 10 * 60 * 1000; // 10 * 60 seconds = 10 minutes

const postAttackDiscoveryRoute = router => {
  router.versioned.post({
    access: 'internal',
    path: _elasticAssistantCommon.ATTACK_DISCOVERY,
    security: {
      authz: {
        requiredPrivileges: ['elasticAssistant']
      }
    },
    options: {
      timeout: {
        idleSocket: ROUTE_HANDLER_TIMEOUT
      }
    }
  }).addVersion({
    version: _elasticAssistantCommon.API_VERSIONS.internal.v1,
    validate: {
      request: {
        body: (0, _common.buildRouteValidationWithZod)(_elasticAssistantCommon.AttackDiscoveryPostRequestBody)
      },
      response: {
        200: {
          body: {
            custom: (0, _common.buildRouteValidationWithZod)(_elasticAssistantCommon.AttackDiscoveryPostResponse)
          }
        }
      }
    }
  }, async (context, request, response) => {
    const performChecksContext = await context.resolve(['core', 'elasticAssistant', 'licensing']);
    const resp = (0, _build_response.buildResponse)(response);
    const assistantContext = await context.elasticAssistant;
    const {
      featureFlags
    } = await context.core;
    const eventLogIndex = (await context.elasticAssistant).eventLogIndex;
    const logger = assistantContext.logger;
    const telemetry = assistantContext.telemetry;
    const savedObjectsClient = assistantContext.savedObjectsClient;
    const checkResponse = await (0, _helpers.performChecks)({
      context: performChecksContext,
      request,
      response
    });
    if (!checkResponse.isSuccess) {
      return checkResponse.response;
    }
    try {
      const eventLogger = (await context.elasticAssistant).eventLogger;

      // get the actions plugin start contract from the request context:
      const actions = (await context.elasticAssistant).actions;
      const actionsClient = await actions.getActionsClientWithRequest(request);
      const dataClient = await assistantContext.getAttackDiscoveryDataClient();
      const authenticatedUser = await assistantContext.getCurrentUser();
      if (authenticatedUser == null) {
        return resp.error({
          body: `Authenticated user not found`,
          statusCode: 401
        });
      }
      if (!dataClient) {
        return resp.error({
          body: `Attack discovery data client not initialized`,
          statusCode: 500
        });
      }

      // get parameters from the request body
      const alertsIndexPattern = decodeURIComponent(request.body.alertsIndexPattern);
      const {
        apiConfig,
        size
      } = request.body;
      if (!(0, _request_is_valid.requestIsValid)({
        alertsIndexPattern,
        request,
        size
      })) {
        return resp.error({
          body: 'Bad Request',
          statusCode: 400
        });
      }

      // get an Elasticsearch client for the authenticated user:
      const esClient = (await context.core).elasticsearch.client.asCurrentUser;
      const attackDiscoveryAlertsEnabled = await featureFlags.getBooleanValue(_elasticAssistantCommon.ATTACK_DISCOVERY_ALERTS_ENABLED_FEATURE_FLAG, true);
      if (attackDiscoveryAlertsEnabled) {
        // Perform alerts access check
        const privilegesCheckResponse = await (0, _index_privileges.hasReadWriteAttackDiscoveryAlertsPrivileges)({
          context: performChecksContext,
          response
        });
        if (!privilegesCheckResponse.isSuccess) {
          return privilegesCheckResponse.response;
        }
      }
      const {
        currentAd,
        attackDiscoveryId
      } = await (0, _helpers2.updateAttackDiscoveryStatusToRunning)(dataClient, authenticatedUser, apiConfig, size);
      const executionUuid = attackDiscoveryAlertsEnabled ? (0, _uuid.v4)() : attackDiscoveryId;

      // event log details:
      const connectorId = apiConfig.connectorId;
      const spaceId = (await context.elasticAssistant).getSpaceId();
      const generatedStarted = new Date();
      const loadingMessage = (0, _elasticAssistantCommon.getAttackDiscoveryLoadingMessage)({
        alertsCount: size,
        end: request.body.end,
        start: request.body.start
      });
      await (0, _write_attack_discovery_event.writeAttackDiscoveryEvent)({
        action: _constants.ATTACK_DISCOVERY_EVENT_LOG_ACTION_GENERATION_STARTED,
        attackDiscoveryAlertsEnabled,
        authenticatedUser,
        connectorId,
        dataClient,
        eventLogger,
        eventLogIndex,
        executionUuid,
        loadingMessage,
        message: `Attack discovery generation ${executionUuid} for user ${authenticatedUser.username} started`,
        spaceId,
        start: generatedStarted
      });

      // Don't await the results of invoking the graph; (just the metadata will be returned from the route handler):
      (0, _generate_and_update_discoveries.generateAndUpdateAttackDiscoveries)({
        actionsClient,
        attackDiscoveryAlertsEnabled,
        executionUuid,
        authenticatedUser,
        config: request.body,
        dataClient,
        esClient,
        logger,
        savedObjectsClient,
        telemetry
      }).then(async result => {
        const end = new Date();
        const durationNanoseconds = (0, _get_duration_nanoseconds.getDurationNanoseconds)({
          end,
          start: generatedStarted
        });

        // NOTE: the (legacy) implementation of generateAttackDiscoveries returns an "error" object when failures occur (instead of rejecting):
        if (result.error == null) {
          var _result$anonymizedAle, _result$attackDiscove, _result$attackDiscove2;
          await (0, _write_attack_discovery_event.writeAttackDiscoveryEvent)({
            action: _constants.ATTACK_DISCOVERY_EVENT_LOG_ACTION_GENERATION_SUCCEEDED,
            alertsContextCount: (_result$anonymizedAle = result.anonymizedAlerts) === null || _result$anonymizedAle === void 0 ? void 0 : _result$anonymizedAle.length,
            attackDiscoveryAlertsEnabled,
            authenticatedUser,
            connectorId,
            dataClient,
            duration: durationNanoseconds,
            end,
            eventLogger,
            eventLogIndex,
            executionUuid,
            message: `Attack discovery generation ${executionUuid} for user ${authenticatedUser.username} succeeded`,
            newAlerts: (_result$attackDiscove = (_result$attackDiscove2 = result.attackDiscoveries) === null || _result$attackDiscove2 === void 0 ? void 0 : _result$attackDiscove2.length) !== null && _result$attackDiscove !== void 0 ? _result$attackDiscove : 0,
            outcome: 'success',
            spaceId
          });
        } else {
          var _result$anonymizedAle2, _result$error, _result$error2;
          await (0, _write_attack_discovery_event.writeAttackDiscoveryEvent)({
            action: _constants.ATTACK_DISCOVERY_EVENT_LOG_ACTION_GENERATION_FAILED,
            alertsContextCount: (_result$anonymizedAle2 = result.anonymizedAlerts) === null || _result$anonymizedAle2 === void 0 ? void 0 : _result$anonymizedAle2.length,
            attackDiscoveryAlertsEnabled,
            authenticatedUser,
            connectorId,
            dataClient,
            duration: durationNanoseconds,
            end,
            eventLogger,
            eventLogIndex,
            executionUuid,
            message: `Attack discovery generation ${executionUuid} for user ${authenticatedUser.username} failed: ${(_result$error = result.error) === null || _result$error === void 0 ? void 0 : _result$error.message}`,
            outcome: 'failure',
            reason: (_result$error2 = result.error) === null || _result$error2 === void 0 ? void 0 : _result$error2.message,
            spaceId
          });
        }
      }).catch(async error => {
        // This is a fallback in case the promise is rejected (in a future implementation):
        const end = new Date();
        const durationNanoseconds = (0, _get_duration_nanoseconds.getDurationNanoseconds)({
          end,
          start: generatedStarted
        });
        await (0, _write_attack_discovery_event.writeAttackDiscoveryEvent)({
          action: _constants.ATTACK_DISCOVERY_EVENT_LOG_ACTION_GENERATION_FAILED,
          attackDiscoveryAlertsEnabled,
          authenticatedUser,
          connectorId,
          dataClient,
          duration: durationNanoseconds,
          end,
          eventLogger,
          eventLogIndex,
          executionUuid,
          message: `Attack discovery generation ${executionUuid} for user ${authenticatedUser.username} failed: ${error.message}`,
          outcome: 'failure',
          reason: error === null || error === void 0 ? void 0 : error.message,
          spaceId
        });
      });
      return response.ok({
        body: currentAd
      });
    } catch (err) {
      logger.error(err);
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return resp.error({
        body: {
          success: false,
          error: error.message
        },
        statusCode: error.statusCode
      });
    }
  });
};
exports.postAttackDiscoveryRoute = postAttackDiscoveryRoute;