"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.extractKey = void 0;
var _es_fields = require("../../common/es_fields");
var _utils = require("../../common/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Extracts the key from the bucket key based on the groupBy fields.
 * If "_index" is not part of the groupBy, bucketKey is returned.
 * Otherwise, it replaces the "_index" value in the bucketKey with the actual dataStream name.
 *
 * @param {Object} params - The parameters object.
 * @param {string[]} params.groupBy - The groupBy fields.
 * @param {string[]} params.bucketKey - The bucket key values.
 * @returns {string[]} The extracted key.
 * @example
 * // returns ['logs-dataset-namespace', 'aws']
 * extractKey({ groupBy: ['_index', 'cloud.provider'], bucketKey: ['.ds-logs-dataset-namespace-2025.04.08-000001', 'aws'] });
 */
const extractKey = ({
  groupBy,
  bucketKey
}) => {
  if (!groupBy.includes(_es_fields.INDEX)) {
    return bucketKey;
  }
  const dataStreamIndex = groupBy.findIndex(group => group === _es_fields.INDEX);
  const dataStreamName = (0, _utils.extractIndexNameFromBackingIndex)(bucketKey[dataStreamIndex]);
  const key = [...bucketKey.slice(0, dataStreamIndex), dataStreamName, ...bucketKey.slice(dataStreamIndex + 1)];
  return key;
};
exports.extractKey = extractKey;