"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getRuleExecutor = exports.formatDurationFromTimeUnitChar = void 0;
var _server = require("@kbn/alerting-plugin/server");
var _i18n = require("@kbn/i18n");
var _utils = require("@kbn/response-ops-rule-params/common/utils");
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _es_fields = require("../../../common/es_fields");
var _context = require("../context");
var _get_docs_stats = require("../get_docs_stats");
var _types = require("../types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const formatDurationFromTimeUnitChar = (time, unit) => {
  const sForPlural = time !== 0 && time > 1 ? 's' : '';
  switch (unit) {
    case 's':
      return `${time} sec${sForPlural}`;
    case 'm':
      return `${time} min${sForPlural}`;
    case 'h':
      return `${time} hr${sForPlural}`;
    case 'd':
      return `${time} day${sForPlural}`;
    default:
      return `${time} ${unit}`;
  }
};
exports.formatDurationFromTimeUnitChar = formatDurationFromTimeUnitChar;
const getRuleExecutor = locatorsClient => async function executor(options) {
  var _params$groupBy;
  const {
    services,
    params,
    logger,
    getTimeRange
  } = options;
  const {
    alertsClient,
    scopedClusterClient
  } = services;
  if (!alertsClient) {
    throw new _server.AlertsClientError();
  }
  const alertLimit = alertsClient.getAlertLimitValue();
  const {
    dateStart,
    dateEnd
  } = getTimeRange(`${params.timeSize}${params.timeUnit}`);
  const index = params.searchConfiguration.index;
  const datasetQualityDegradedResults = await (0, _get_docs_stats.getDocsStats)({
    index,
    dateStart,
    groupBy: (_params$groupBy = params.groupBy) !== null && _params$groupBy !== void 0 ? _params$groupBy : [],
    query: {
      must: {
        exists: {
          field: _es_fields._IGNORED
        }
      }
    },
    scopedClusterClient
  });
  const unmetGroupValues = {};
  const compareFn = _utils.ComparatorFns.get(params.comparator);
  if (compareFn == null) {
    throw new Error(_i18n.i18n.translate('xpack.datasetQuality.rule.invalidComparatorErrorMessage', {
      defaultMessage: 'invalid thresholdComparator specified: {comparator}',
      values: {
        comparator: params.comparator
      }
    }));
  }
  let generatedAlerts = 0;
  for (const groupResult of datasetQualityDegradedResults) {
    var _params$groupBy2;
    const {
      bucketKey,
      percentage
    } = groupResult;
    const alertId = bucketKey.join(_ruleDataUtils.ALERT_KEY_JOINER);
    const met = compareFn(percentage, params.threshold);
    if (!met) {
      unmetGroupValues[alertId] = percentage.toFixed(2);
      continue;
    }
    const groupByFields = {};
    const groupBy = (_params$groupBy2 = params.groupBy) !== null && _params$groupBy2 !== void 0 ? _params$groupBy2 : [];
    for (let i = 0; i < bucketKey.length; i++) {
      const fieldName = groupBy[i];
      groupByFields[fieldName] = bucketKey[i];
    }
    if (generatedAlerts < alertLimit) {
      var _params$threshold;
      const context = (0, _context.generateContext)({
        group: alertId,
        dateStart,
        dateEnd,
        value: percentage.toFixed(2),
        params,
        grouping: groupByFields,
        locatorsClient
      });
      alertsClient.report({
        id: alertId,
        actionGroup: _types.THRESHOLD_MET_GROUP.id,
        state: {},
        context,
        payload: {
          [_ruleDataUtils.ALERT_REASON]: context.reason,
          [_ruleDataUtils.ALERT_EVALUATION_VALUE]: `${context.value}`,
          [_ruleDataUtils.ALERT_EVALUATION_THRESHOLD]: ((_params$threshold = params.threshold) === null || _params$threshold === void 0 ? void 0 : _params$threshold.length) === 1 ? params.threshold[0] : null,
          [_ruleDataUtils.ALERT_GROUPING]: groupByFields
        }
      });
    }
    generatedAlerts++;
  }
  alertsClient.setAlertLimitReached(generatedAlerts >= alertLimit);

  // Handle recovered alerts context
  const {
    getRecoveredAlerts
  } = alertsClient;
  for (const recoveredAlert of getRecoveredAlerts()) {
    var _recoveredAlert$hit, _unmetGroupValues$ale, _params$threshold2;
    const alertId = recoveredAlert.alert.getId();
    logger.debug(`setting context for recovered alert ${alertId}`);
    const grouping = (_recoveredAlert$hit = recoveredAlert.hit) === null || _recoveredAlert$hit === void 0 ? void 0 : _recoveredAlert$hit[_ruleDataUtils.ALERT_GROUPING];
    const percentage = (_unmetGroupValues$ale = unmetGroupValues[alertId]) !== null && _unmetGroupValues$ale !== void 0 ? _unmetGroupValues$ale : '0';
    const recoveryContext = (0, _context.generateContext)({
      group: alertId,
      dateStart,
      dateEnd,
      value: percentage,
      params,
      grouping,
      locatorsClient
    });
    alertsClient.setAlertData({
      id: alertId,
      context: recoveryContext,
      payload: {
        [_ruleDataUtils.ALERT_REASON]: recoveryContext.reason,
        [_ruleDataUtils.ALERT_EVALUATION_VALUE]: `${recoveryContext.value}`,
        [_ruleDataUtils.ALERT_EVALUATION_THRESHOLD]: ((_params$threshold2 = params.threshold) === null || _params$threshold2 === void 0 ? void 0 : _params$threshold2.length) === 1 ? params.threshold[0] : null,
        [_ruleDataUtils.ALERT_GROUPING]: grouping
      }
    });
  }
  return {
    state: {}
  };
};
exports.getRuleExecutor = getRuleExecutor;