"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.datasetQualityEbtEvents = void 0;
var _lodash = require("lodash");
var _types = require("./types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const dataStreamSchema = {
  properties: {
    dataset: {
      type: 'keyword',
      _meta: {
        description: 'Data stream dataset name'
      }
    },
    namespace: {
      type: 'keyword',
      _meta: {
        description: 'Data stream namespace'
      }
    },
    type: {
      type: 'keyword',
      _meta: {
        description: 'Data stream type e.g. "logs", "metrics"'
      }
    }
  }
};
const privilegesSchema = {
  properties: {
    can_monitor_data_stream: {
      type: 'boolean',
      _meta: {
        description: 'Whether user can monitor the data stream'
      }
    },
    can_view_integrations: {
      type: 'boolean',
      _meta: {
        description: 'Whether user can view integrations'
      }
    },
    can_view_dashboards: {
      type: 'boolean',
      _meta: {
        description: 'Whether user can view dashboards',
        optional: true
      }
    }
  }
};
const ebtFilterObjectSchema = {
  properties: {
    total: {
      type: 'short',
      _meta: {
        description: 'Total number of values available to filter',
        optional: false
      }
    },
    included: {
      type: 'short',
      _meta: {
        description: 'Number of values selected to filter for',
        optional: false
      }
    },
    excluded: {
      type: 'short',
      _meta: {
        description: 'Number of values selected to filter out',
        optional: false
      }
    }
  },
  _meta: {
    description: 'Represents the multi select filters',
    optional: false
  }
};
const sortSchema = {
  properties: {
    field: {
      type: 'keyword',
      _meta: {
        description: 'Field used for sorting on the main table',
        optional: false
      }
    },
    direction: {
      type: 'keyword',
      _meta: {
        description: 'Sort direction',
        optional: false
      }
    }
  },
  _meta: {
    description: 'Represents the state of applied sorting on the dataset quality home page',
    optional: false
  }
};
const filtersSchema = {
  properties: {
    is_degraded: {
      type: 'boolean',
      _meta: {
        description: 'Whether _ignored filter is applied',
        optional: false
      }
    },
    query_length: {
      type: 'short',
      _meta: {
        description: 'Length of the query string',
        optional: false
      }
    },
    integrations: ebtFilterObjectSchema,
    namespaces: ebtFilterObjectSchema,
    qualities: ebtFilterObjectSchema
  },
  _meta: {
    description: 'Represents the state of applied filters on the dataset quality home page',
    optional: false
  }
};
const datasetCommonSchema = {
  index_name: {
    type: 'keyword',
    _meta: {
      description: 'Index name'
    }
  },
  data_stream: dataStreamSchema,
  privileges: privilegesSchema,
  data_stream_health: {
    type: 'keyword',
    _meta: {
      description: 'Quality of the data stream e.g. "good", "degraded", "poor"'
    }
  },
  data_stream_aggregatable: {
    type: 'boolean',
    _meta: {
      description: 'Whether data stream is aggregatable against _ignored field'
    }
  },
  degraded_percentage: {
    type: 'float',
    _meta: {
      description: 'Percentage of degraded documents in the data stream'
    }
  },
  from: {
    type: 'date',
    _meta: {
      description: 'Start of the time range ISO8601 formatted string'
    }
  },
  to: {
    type: 'date',
    _meta: {
      description: 'End of the time range ISO8601 formatted string'
    }
  },
  integration: {
    type: 'keyword',
    _meta: {
      description: 'Integration name, if any',
      optional: true
    }
  }
};
const datasetNavigatedEventType = {
  eventType: _types.DatasetQualityTelemetryEventTypes.NAVIGATED,
  schema: {
    ...datasetCommonSchema,
    sort: sortSchema,
    filters: filtersSchema,
    target: {
      type: 'keyword',
      _meta: {
        description: 'Action that user took to navigate away from the dataset quality page'
      }
    },
    source: {
      type: 'keyword',
      _meta: {
        description: 'Section of dataset quality page the action is originated from e.g. header, summary, chart or table etc.'
      }
    }
  }
};
const datasetDetailsOpenedEventType = {
  eventType: _types.DatasetQualityTelemetryEventTypes.DETAILS_OPENED,
  schema: {
    ...datasetCommonSchema,
    tracking_id: {
      type: 'keyword',
      _meta: {
        description: `Locally generated session tracking ID for funnel analysis`
      }
    },
    duration: {
      type: 'long',
      _meta: {
        description: 'Duration in milliseconds to load the dataset details page'
      }
    },
    breakdown_field: {
      type: 'keyword',
      _meta: {
        description: 'Field used for chart breakdown, if any',
        optional: true
      }
    }
  }
};
const datasetDetailsNavigatedEventType = {
  eventType: _types.DatasetQualityTelemetryEventTypes.DETAILS_NAVIGATED,
  schema: {
    ...(0, _lodash.omit)(datasetDetailsOpenedEventType.schema, 'duration'),
    filters: {
      properties: {
        is_degraded: {
          type: 'boolean',
          _meta: {
            description: 'Whether _ignored filter is applied to the link',
            optional: false
          }
        }
      }
    },
    target: {
      type: 'keyword',
      _meta: {
        description: 'Action that user took to navigate away from the dataset details page'
      }
    },
    source: {
      type: 'keyword',
      _meta: {
        description: 'Section of dataset details page the action is originated from e.g. header, summary, chart or table etc.'
      }
    }
  }
};
const datasetDetailsBreakdownFieldChangedEventType = {
  eventType: _types.DatasetQualityTelemetryEventTypes.BREAKDOWN_FIELD_CHANGED,
  schema: {
    ...datasetCommonSchema,
    tracking_id: {
      type: 'keyword',
      _meta: {
        description: `Locally generated session tracking ID for funnel analysis`
      }
    },
    breakdown_field: {
      type: 'keyword',
      _meta: {
        description: 'Field used for chart breakdown, if any',
        optional: true
      }
    }
  }
};
const datasetQualityEbtEvents = exports.datasetQualityEbtEvents = {
  datasetNavigatedEventType,
  datasetDetailsOpenedEventType,
  datasetDetailsNavigatedEventType,
  datasetDetailsBreakdownFieldChangedEventType
};