"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getFieldExamplesRequest = exports.fetchFieldsExamples = void 0;
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _mlIsPopulatedObject = require("@kbn/ml-is-populated-object");
var _mlErrorUtils = require("@kbn/ml-error-utils");
var _example_utils = require("../../../common/util/example_utils");
var _field_stats = require("../../../../../common/types/field_stats");
var _constants = require("./constants");
var _build_query_filters = require("../../../../../common/utils/build_query_filters");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getFieldExamplesRequest = (params, field) => {
  const {
    index,
    timeFieldName,
    earliestMs,
    latestMs,
    query,
    runtimeFieldMap,
    maxExamples
  } = params;

  // Request at least 100 docs so that we have a chance of obtaining
  // 'maxExamples' of the field.
  const size = Math.max(100, maxExamples !== null && maxExamples !== void 0 ? maxExamples : _constants.MAX_EXAMPLES_DEFAULT);
  const filterCriteria = (0, _build_query_filters.buildFilterCriteria)(timeFieldName, earliestMs, latestMs, query);

  // Use an exists filter to return examples of the field.
  if (Array.isArray(filterCriteria)) {
    filterCriteria.push({
      exists: {
        field: field.fieldName
      }
    });
  }
  const searchBody = {
    fields: [field.fieldName],
    _source: false,
    query: {
      bool: {
        filter: filterCriteria
      }
    },
    ...((0, _mlIsPopulatedObject.isPopulatedObject)(runtimeFieldMap) ? {
      runtime_mappings: runtimeFieldMap
    } : {})
  };
  return {
    index,
    size,
    body: searchBody
  };
};
exports.getFieldExamplesRequest = getFieldExamplesRequest;
const fetchFieldsExamples = (dataSearch, params, fields, options) => {
  const {
    maxExamples
  } = params;
  return (0, _rxjs.combineLatest)(fields.map(field => {
    const request = getFieldExamplesRequest(params, field);
    return dataSearch.search({
      params: request
    }, options).pipe((0, _rxjs.catchError)(e => (0, _rxjs.of)({
      fieldName: field.fieldName,
      fields,
      error: (0, _mlErrorUtils.extractErrorProperties)(e)
    })), (0, _rxjs.map)(resp => {
      if (!(0, _field_stats.isIKibanaSearchResponse)(resp)) return resp;
      const body = resp.rawResponse;
      const stats = {
        fieldName: field.fieldName,
        examples: []
      };
      if (body.hits.total > 0) {
        const hits = body.hits.hits;
        const processedDocs = hits.map(hit => {
          const doc = (0, _lodash.get)(hit.fields, field.fieldName);
          return Array.isArray(doc) && doc.length > 0 ? doc[0] : doc;
        });
        stats.examples = (0, _example_utils.getUniqGeoOrStrExamples)(processedDocs, maxExamples);
      }
      return stats;
    }));
  }));
};
exports.fetchFieldsExamples = fetchFieldsExamples;