"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SearchSessionsMgmtAPI = void 0;
var _i18n = require("@kbn/i18n");
var _moment = _interopRequireDefault(require("moment"));
var _rxjs = require("rxjs");
var _actions = require("../components/actions");
var _common = require("../../../../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function getActions(status) {
  const actions = [];
  actions.push(_actions.ACTION.INSPECT);
  actions.push(_actions.ACTION.RENAME);
  if (status === _common.SearchSessionStatus.IN_PROGRESS || status === _common.SearchSessionStatus.COMPLETE) {
    actions.push(_actions.ACTION.EXTEND);
    actions.push(_actions.ACTION.DELETE);
  }
  if (status === _common.SearchSessionStatus.EXPIRED || status === _common.SearchSessionStatus.ERROR || status === _common.SearchSessionStatus.CANCELLED) {
    actions.push(_actions.ACTION.DELETE);
  }
  return actions;
}
function getUrlFromState(locators, locatorId, state) {
  try {
    const locator = locators.get(locatorId);
    return locator === null || locator === void 0 ? void 0 : locator.getRedirectUrl(state);
  } catch (err) {
    // eslint-disable-next-line no-console
    console.error('Could not create URL from restoreState');
    // eslint-disable-next-line no-console
    console.error(err);
  }
}

// Helper: factory for a function to map server objects to UI objects
const mapToUISession = (locators, config, sessionStatuses) => async savedObject => {
  var _sessionStatuses$save, _sessionStatuses$save2;
  const {
    name,
    appId,
    created,
    expires,
    locatorId,
    initialState,
    restoreState,
    idMapping,
    version
  } = savedObject.attributes;
  const status = (_sessionStatuses$save = sessionStatuses[savedObject.id]) === null || _sessionStatuses$save === void 0 ? void 0 : _sessionStatuses$save.status;
  const errors = (_sessionStatuses$save2 = sessionStatuses[savedObject.id]) === null || _sessionStatuses$save2 === void 0 ? void 0 : _sessionStatuses$save2.errors;
  const actions = getActions(status);

  // TODO: initialState should be saved without the searchSessionID
  if (initialState) delete initialState.searchSessionId;
  // derive the URL and add it in
  const reloadUrl = await getUrlFromState(locators, locatorId, initialState);
  const restoreUrl = await getUrlFromState(locators, locatorId, restoreState);
  return {
    id: savedObject.id,
    name,
    appId,
    created,
    expires,
    status,
    actions,
    restoreUrl: restoreUrl,
    reloadUrl: reloadUrl,
    initialState,
    restoreState,
    idMapping,
    numSearches: Object.keys(idMapping).length,
    version,
    errors
  };
};
class SearchSessionsMgmtAPI {
  constructor(sessionsClient, config, deps) {
    this.sessionsClient = sessionsClient;
    this.config = config;
    this.deps = deps;
  }
  async fetchTableData() {
    const mgmtConfig = this.config.management;
    const refreshTimeout = _moment.default.duration(mgmtConfig.refreshTimeout);
    const fetch$ = (0, _rxjs.from)(this.sessionsClient.find({
      page: 1,
      perPage: mgmtConfig.maxSessions,
      sortField: 'created',
      sortOrder: 'desc'
    }));
    const timeout$ = (0, _rxjs.timer)(refreshTimeout.asMilliseconds()).pipe((0, _rxjs.tap)(() => {
      this.deps.notifications.toasts.addDanger(_i18n.i18n.translate('data.mgmt.searchSessions.api.fetchTimeout', {
        defaultMessage: 'Fetching the Search Session info timed out after {timeout} seconds',
        values: {
          timeout: refreshTimeout.asSeconds()
        }
      }));
    }), (0, _rxjs.mapTo)(null));

    // fetch the search sessions before timeout triggers
    try {
      const result = await (0, _rxjs.race)(fetch$, timeout$).toPromise();
      if (result && result.saved_objects) {
        const savedObjects = result.saved_objects;
        return await Promise.all(savedObjects.map(mapToUISession(this.deps.locators, this.config, result.statuses)));
      }
    } catch (err) {
      // eslint-disable-next-line no-console
      console.error(err);
      this.deps.notifications.toasts.addError(err, {
        title: _i18n.i18n.translate('data.mgmt.searchSessions.api.fetchError', {
          defaultMessage: 'Failed to refresh the page!'
        })
      });
    }
    return [];
  }
  getExtendByDuration() {
    return this.config.defaultExpiration;
  }

  // Delete and expire
  async sendDelete(id) {
    var _this$deps$usageColle;
    (_this$deps$usageColle = this.deps.usageCollector) === null || _this$deps$usageColle === void 0 ? void 0 : _this$deps$usageColle.trackSessionDeleted();
    try {
      await this.sessionsClient.delete(id);
      this.deps.notifications.toasts.addSuccess({
        title: _i18n.i18n.translate('data.mgmt.searchSessions.api.deleted', {
          defaultMessage: 'The search session was deleted.'
        })
      });
    } catch (err) {
      this.deps.notifications.toasts.addError(err, {
        title: _i18n.i18n.translate('data.mgmt.searchSessions.api.deletedError', {
          defaultMessage: 'Failed to delete the search session!'
        })
      });
    }
  }

  // Extend
  async sendExtend(id, expires) {
    var _this$deps$usageColle2;
    (_this$deps$usageColle2 = this.deps.usageCollector) === null || _this$deps$usageColle2 === void 0 ? void 0 : _this$deps$usageColle2.trackSessionExtended();
    try {
      await this.sessionsClient.extend(id, expires);
      this.deps.notifications.toasts.addSuccess({
        title: _i18n.i18n.translate('data.mgmt.searchSessions.api.extended', {
          defaultMessage: 'The search session was extended.'
        })
      });
    } catch (err) {
      this.deps.notifications.toasts.addError(err, {
        title: _i18n.i18n.translate('data.mgmt.searchSessions.api.extendError', {
          defaultMessage: 'Failed to extend the search session!'
        })
      });
    }
  }

  // Change the user-facing name of a search session
  async sendRename(id, newName) {
    try {
      await this.sessionsClient.rename(id, newName);
      this.deps.notifications.toasts.addSuccess({
        title: _i18n.i18n.translate('data.mgmt.searchSessions.api.rename', {
          defaultMessage: 'The search session was renamed'
        })
      });
    } catch (err) {
      this.deps.notifications.toasts.addError(err, {
        title: _i18n.i18n.translate('data.mgmt.searchSessions.api.renameError', {
          defaultMessage: 'Failed to rename the search session'
        })
      });
    }
  }
}
exports.SearchSessionsMgmtAPI = SearchSessionsMgmtAPI;