"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CloudExperimentsPlugin = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _moment = require("moment");
var _launchdarklyClientProvider = require("@openfeature/launchdarkly-client-provider");
var _launchdarklyJsClientSdk = require("launchdarkly-js-client-sdk");
var _metadata_service = require("../common/metadata_service");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Browser-side implementation of the Cloud Experiments plugin
 */
class CloudExperimentsPlugin {
  /** Constructor of the plugin **/
  constructor(initializerContext) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "metadataService", void 0);
    this.initializerContext = initializerContext;
    this.logger = initializerContext.logger.get();
    const config = initializerContext.config.get();
    this.metadataService = new _metadata_service.MetadataService({
      metadata_refresh_interval: (0, _moment.duration)(config.metadata_refresh_interval)
    }, this.logger.get('metadata'));
    const ldConfig = config.launch_darkly;
    if (!(ldConfig !== null && ldConfig !== void 0 && ldConfig.client_id) && !initializerContext.env.mode.dev) {
      // If the plugin is enabled, and it's in prod mode, launch_darkly must exist
      // (config-schema should enforce it, but just in case).
      throw new Error('xpack.cloud_integrations.experiments.launch_darkly configuration should exist');
    }
  }

  /**
   * Sets up the A/B testing client only if cloud is enabled
   * @param core {@link CoreSetup}
   * @param deps {@link CloudExperimentsPluginSetupDeps}
   */
  setup(core, deps) {
    (0, _metadata_service.initializeMetadata)({
      metadataService: this.metadataService,
      initializerContext: this.initializerContext,
      cloud: deps.cloud,
      featureFlags: core.featureFlags,
      logger: this.logger
    });
    const initialFeatureFlags = core.featureFlags.getInitialFeatureFlags();
    const launchDarklyOpenFeatureProvider = this.createOpenFeatureProvider(initialFeatureFlags);
    if (launchDarklyOpenFeatureProvider) {
      core.featureFlags.setProvider(launchDarklyOpenFeatureProvider);
    }
  }

  /**
   * Sets the metadata service update hooks
   * @param core {@link CoreStart}
   * @param deps {@link CloudExperimentsPluginStartDeps}
   */
  start(core, {
    dataViews
  }) {
    this.metadataService.start({
      hasDataFetcher: async () => ({
        has_data: await dataViews.hasData.hasUserDataView()
      })
    });
  }

  /**
   * Cleans up and flush the sending queues.
   */
  stop() {
    this.metadataService.stop();
  }

  /**
   * Sets up the OpenFeature LaunchDarkly provider
   * @internal
   */
  createOpenFeatureProvider(initialFeatureFlags) {
    const {
      launch_darkly: ldConfig
    } = this.initializerContext.config.get();
    if (!ldConfig) return;
    const bootstrap = Object.keys(initialFeatureFlags).length > 0 ? initialFeatureFlags : undefined;
    return new _launchdarklyClientProvider.LaunchDarklyClientProvider(ldConfig.client_id, {
      // logger: this.logger.get('launch-darkly'),
      // Using basicLogger for now because we can't limit the level for now if we're using core's logger.
      logger: (0, _launchdarklyJsClientSdk.basicLogger)({
        level: ldConfig.client_log_level
      }),
      streaming: true,
      // Necessary to react to flag changes
      application: {
        id: 'kibana-browser',
        version: this.initializerContext.env.packageInfo.buildFlavor === 'serverless' ? this.initializerContext.env.packageInfo.buildSha : this.initializerContext.env.packageInfo.version
      },
      bootstrap
    });
  }
}
exports.CloudExperimentsPlugin = CloudExperimentsPlugin;