"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getCanvasWorkpadService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lib = require("../../common/lib");
var _kibana_services = require("./kibana_services");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/*
  Remove any top level keys from the workpad which will be rejected by validation
*/
const validKeys = ['@created', '@timestamp', 'assets', 'colors', 'css', 'variables', 'height', 'id', 'isWriteable', 'name', 'page', 'pages', 'width'];
const sanitizeWorkpad = function (workpad) {
  const workpadKeys = Object.keys(workpad);
  for (const key of workpadKeys) {
    if (!validKeys.includes(key)) {
      delete workpad[key];
    }
  }
  return workpad;
};
class CanvasWorkpadService {
  constructor() {
    (0, _defineProperty2.default)(this, "apiPath", `${_lib.API_ROUTE_WORKPAD}`);
  }
  async get(id) {
    const workpad = await _kibana_services.coreServices.http.get(`${this.apiPath}/${id}`, {
      version: '1'
    });
    return {
      css: _lib.DEFAULT_WORKPAD_CSS,
      variables: [],
      ...workpad
    };
  }
  async export(id) {
    var _attributes$css, _attributes$variables;
    const workpad = await _kibana_services.coreServices.http.get(`${this.apiPath}/export/${id}`, {
      version: '1'
    });
    const {
      attributes
    } = workpad;
    return {
      ...workpad,
      attributes: {
        ...attributes,
        css: (_attributes$css = attributes.css) !== null && _attributes$css !== void 0 ? _attributes$css : _lib.DEFAULT_WORKPAD_CSS,
        variables: (_attributes$variables = attributes.variables) !== null && _attributes$variables !== void 0 ? _attributes$variables : []
      }
    };
  }
  async resolve(id) {
    const {
      workpad,
      ...resolveProps
    } = await _kibana_services.coreServices.http.get(`${this.apiPath}/resolve/${id}`, {
      version: '1'
    });
    return {
      ...resolveProps,
      workpad: {
        // @ts-ignore: Shimming legacy workpads that might not have CSS
        css: _lib.DEFAULT_WORKPAD_CSS,
        // @ts-ignore: Shimming legacy workpads that might not have variables
        variables: [],
        ...workpad
      }
    };
  }
  async create(workpad) {
    return _kibana_services.coreServices.http.post(this.apiPath, {
      body: JSON.stringify({
        ...sanitizeWorkpad({
          ...workpad
        }),
        assets: workpad.assets || {},
        variables: workpad.variables || []
      }),
      version: '1'
    });
  }
  async import(workpad) {
    return _kibana_services.coreServices.http.post(`${this.apiPath}/import`, {
      body: JSON.stringify({
        ...sanitizeWorkpad({
          ...workpad
        }),
        assets: workpad.assets || {},
        variables: workpad.variables || []
      }),
      version: '1'
    });
  }
  async createFromTemplate(templateId) {
    return _kibana_services.coreServices.http.post(this.apiPath, {
      body: JSON.stringify({
        templateId
      }),
      version: '1'
    });
  }
  async findTemplates() {
    return _kibana_services.coreServices.http.get(_lib.API_ROUTE_TEMPLATES, {
      version: '1'
    });
  }
  async find(searchTerm) {
    // TODO: this shouldn't be necessary.  Check for usage.
    const validSearchTerm = typeof searchTerm === 'string' && searchTerm.length > 0;
    return _kibana_services.coreServices.http.get(`${this.apiPath}/find`, {
      query: {
        perPage: 10000,
        name: validSearchTerm ? searchTerm : ''
      },
      version: '1'
    });
  }
  async remove(id) {
    _kibana_services.coreServices.http.delete(`${this.apiPath}/${id}`, {
      version: '1'
    });
  }
  async update(id, workpad) {
    _kibana_services.coreServices.http.put(`${this.apiPath}/${id}`, {
      body: JSON.stringify({
        ...sanitizeWorkpad({
          ...workpad
        })
      }),
      version: '1'
    });
  }
  async updateWorkpad(id, workpad) {
    _kibana_services.coreServices.http.put(`${_lib.API_ROUTE_WORKPAD_STRUCTURES}/${id}`, {
      body: JSON.stringify({
        ...sanitizeWorkpad({
          ...workpad
        })
      }),
      version: '1'
    });
  }
  async updateAssets(id, assets) {
    _kibana_services.coreServices.http.put(`${_lib.API_ROUTE_WORKPAD_ASSETS}/${id}`, {
      body: JSON.stringify(assets),
      version: '1'
    });
  }
  async getRuntimeZip(workpad) {
    return _kibana_services.coreServices.http.post(_lib.API_ROUTE_SHAREABLE_ZIP, {
      body: JSON.stringify(workpad),
      version: '1'
    });
  }
}
let canvasWorkpadService;
const getCanvasWorkpadService = () => {
  if (!canvasWorkpadService) {
    canvasWorkpadService = new CanvasWorkpadService();
  }
  return canvasWorkpadService;
};
exports.getCanvasWorkpadService = getCanvasWorkpadService;