"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TraceDataState = void 0;
exports.getClockSkew = getClockSkew;
exports.getRootItemOrFallback = getRootItemOrFallback;
exports.getServiceColors = getServiceColors;
exports.getServiceLegends = getServiceLegends;
exports.getTraceParentChildrenMap = getTraceParentChildrenMap;
exports.getTraceWaterfall = getTraceWaterfall;
exports.getTraceWaterfallDuration = getTraceWaterfallDuration;
exports.useTraceWaterfall = useTraceWaterfall;
var _eui = require("@elastic/eui");
var _react = require("react");
var _i18n = require("@kbn/i18n");
var _waterfall_helpers = require("../../app/transaction_details/waterfall_with_summary/waterfall_container/waterfall/waterfall_helpers/waterfall_helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const FALLBACK_WARNING = _i18n.i18n.translate('xpack.apm.traceWaterfallItem.warningMessage.fallbackWarning', {
  defaultMessage: 'The trace document is incomplete and not all spans have arrived yet. Try refreshing the page or adjusting the time range.'
});
const INSTRUMENTATION_WARNING = _i18n.i18n.translate('xpack.apm.traceWaterfallItem.euiCallOut.aDuplicatedSpanWasLabel', {
  defaultMessage: 'A duplicated span was detected. This indicates a problem with how your services have been instrumented, as span IDs are meant to be unique.'
});
function useTraceWaterfall({
  traceItems
}) {
  const waterfall = (0, _react.useMemo)(() => {
    try {
      const serviceColorsMap = getServiceColors(traceItems);
      const traceParentChildrenMap = getTraceParentChildrenMap(traceItems);
      const {
        rootItem,
        traceState,
        orphans
      } = getRootItemOrFallback(traceParentChildrenMap, traceItems);
      const traceWaterfall = rootItem ? getTraceWaterfall({
        rootItem,
        parentChildMap: traceParentChildrenMap,
        orphans,
        serviceColorsMap
      }) : [];
      return {
        rootItem,
        traceState,
        message: traceState !== TraceDataState.Full ? FALLBACK_WARNING : undefined,
        traceWaterfall,
        duration: getTraceWaterfallDuration(traceWaterfall),
        maxDepth: Math.max(...traceWaterfall.map(item => item.depth))
      };
    } catch (e) {
      return {
        traceState: TraceDataState.Invalid,
        message: INSTRUMENTATION_WARNING,
        traceWaterfall: [],
        duration: 0,
        maxDepth: 0
      };
    }
  }, [traceItems]);
  return waterfall;
}
function getServiceColors(traceItems) {
  const allServiceNames = new Set(traceItems.map(item => item.serviceName));
  const palette = (0, _eui.euiPaletteColorBlind)({
    rotations: Math.ceil(allServiceNames.size / 10)
  });
  return Array.from(allServiceNames).reduce((acc, serviceName, idx) => {
    acc[serviceName] = palette[idx];
    return acc;
  }, {});
}
function getServiceLegends(traceItems) {
  const allServiceNames = new Set(traceItems.map(item => item.serviceName));
  const palette = (0, _eui.euiPaletteColorBlind)({
    rotations: Math.ceil(allServiceNames.size / 10)
  });
  return Array.from(allServiceNames).map((serviceName, index) => ({
    type: _waterfall_helpers.WaterfallLegendType.ServiceName,
    value: serviceName,
    color: palette[index]
  }));
}
function getTraceParentChildrenMap(traceItems) {
  const traceMap = traceItems.reduce((acc, item) => {
    if (!item.parentId) {
      acc.root = [item];
    } else {
      if (!acc[item.parentId]) {
        acc[item.parentId] = [];
      }
      acc[item.parentId].push(item);
    }
    return acc;
  }, {});
  return traceMap;
}
let TraceDataState = exports.TraceDataState = /*#__PURE__*/function (TraceDataState) {
  TraceDataState["Full"] = "full";
  TraceDataState["Partial"] = "partial";
  TraceDataState["Empty"] = "empty";
  TraceDataState["Invalid"] = "invalid";
  return TraceDataState;
}({});
function getRootItemOrFallback(traceParentChildrenMap, traceItems) {
  var _traceParentChildrenM;
  if (traceItems.length === 0) {
    return {
      traceState: TraceDataState.Empty
    };
  }
  const rootItem = (_traceParentChildrenM = traceParentChildrenMap.root) === null || _traceParentChildrenM === void 0 ? void 0 : _traceParentChildrenM[0];
  const parentIds = new Set(traceItems.map(({
    id
  }) => id));
  // TODO: Reuse waterfall util methods where possible or if logic is the same
  const orphans = traceItems.filter(item => item.parentId && !parentIds.has(item.parentId));
  if (rootItem) {
    return {
      traceState: orphans.length === 0 ? TraceDataState.Full : TraceDataState.Partial,
      rootItem,
      orphans
    };
  }
  const [fallbackRootItem, ...remainingOrphans] = orphans;
  return {
    traceState: TraceDataState.Partial,
    rootItem: fallbackRootItem,
    orphans: remainingOrphans
  };
}

// TODO: Reuse waterfall util methods where possible or if logic is the same
function reparentOrphansToRoot(rootItem, parentChildMap, orphans) {
  // Some cases with orphans, the root item has no direct link or children, so this
  // might be not initialised. This assigns the array in case of undefined/null to the
  // map.
  const children = parentChildMap[rootItem.id] ??= [];
  children.push(...orphans.map(orphan => ({
    ...orphan,
    parentId: rootItem.id,
    isOrphan: true
  })));
}
function getTraceWaterfall({
  rootItem,
  parentChildMap,
  orphans,
  serviceColorsMap
}) {
  const rootStartMicroseconds = rootItem.timestampUs;
  const visitor = new Set([rootItem.id]);
  reparentOrphansToRoot(rootItem, parentChildMap, orphans);
  function getTraceWaterfallItem(item, depth, parent) {
    var _parentChildMap$item$;
    const startMicroseconds = item.timestampUs;
    const traceWaterfallItem = {
      ...item,
      depth,
      offset: startMicroseconds - rootStartMicroseconds,
      skew: getClockSkew({
        itemTimestamp: startMicroseconds,
        itemDuration: item.duration,
        parent
      }),
      color: serviceColorsMap[item.serviceName]
    };
    const sortedChildren = ((_parentChildMap$item$ = parentChildMap[item.id]) === null || _parentChildMap$item$ === void 0 ? void 0 : _parentChildMap$item$.sort((a, b) => a.timestampUs - b.timestampUs)) || [];
    const flattenedChildren = sortedChildren.flatMap(child => {
      // Check if we have encountered the trace item before.
      // If we have visited the trace item before, then the child waterfall items are already
      // present in the flattened list, so we throw an error to alert the user of duplicated
      // spans. This should guard against circular or unusual links between spans.
      if (visitor.has(child.id)) {
        throw new Error('Duplicate span id detected');
      }

      // If we haven't visited it before, then we can process the waterfall item.
      visitor.add(child.id);
      return getTraceWaterfallItem(child, depth + 1, traceWaterfallItem);
    });
    return [traceWaterfallItem, ...flattenedChildren];
  }
  return getTraceWaterfallItem(rootItem, 0);
}
function getClockSkew({
  itemTimestamp,
  itemDuration,
  parent
}) {
  let skew = 0;
  if (parent) {
    const parentTimestamp = parent.timestampUs;
    const parentStart = parentTimestamp + parent.skew;
    const offsetStart = parentStart - itemTimestamp;
    if (offsetStart > 0) {
      const latency = Math.max(parent.duration - itemDuration, 0) / 2;
      skew = offsetStart + latency;
    }
  }
  return skew;
}
function getTraceWaterfallDuration(flattenedTraceWaterfall) {
  return Math.max(...flattenedTraceWaterfall.map(item => item.offset + item.skew + item.duration), 0);
}