"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.dayAndWeekBeforeToOffset = exports.TimeRangeComparisonEnum = void 0;
exports.getComparisonOptions = getComparisonOptions;
exports.isDefined = isDefined;
exports.isTimeComparison = exports.isExpectedBoundsComparison = void 0;
var _moment = _interopRequireDefault(require("moment"));
var _i18n = require("@kbn/i18n");
var _get_offset_in_ms = require("../../../../common/utils/get_offset_in_ms");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
let TimeRangeComparisonEnum = exports.TimeRangeComparisonEnum = /*#__PURE__*/function (TimeRangeComparisonEnum) {
  TimeRangeComparisonEnum["WeekBefore"] = "week";
  TimeRangeComparisonEnum["DayBefore"] = "day";
  TimeRangeComparisonEnum["PeriodBefore"] = "period";
  TimeRangeComparisonEnum["ExpectedBounds"] = "expected_bounds";
  return TimeRangeComparisonEnum;
}({});
const isTimeComparison = v => v !== TimeRangeComparisonEnum.ExpectedBounds;
exports.isTimeComparison = isTimeComparison;
const isExpectedBoundsComparison = v => v === TimeRangeComparisonEnum.ExpectedBounds;
exports.isExpectedBoundsComparison = isExpectedBoundsComparison;
const dayAndWeekBeforeToOffset = exports.dayAndWeekBeforeToOffset = {
  [TimeRangeComparisonEnum.DayBefore]: '1d',
  [TimeRangeComparisonEnum.WeekBefore]: '1w'
};
const twentyFiveHoursInMs = _moment.default.duration(25, 'h').asMilliseconds();
const eightDaysInMs = _moment.default.duration(8, 'd').asMilliseconds();
function formatDate({
  currentPeriodEnd,
  previousPeriodStart,
  previousPeriodEnd
}) {
  const isDifferentYears = previousPeriodStart.get('year') !== currentPeriodEnd.get('year');
  const dateFormat = isDifferentYears ? 'DD/MM/YY HH:mm' : 'DD/MM HH:mm';
  return `${previousPeriodStart.format(dateFormat)} - ${previousPeriodEnd.format(dateFormat)}`;
}
function isDefined(argument) {
  return argument !== undefined && argument !== null;
}
function getSelectOptions({
  comparisonTypes,
  start,
  end,
  msDiff
}) {
  return comparisonTypes.map(value => {
    switch (value) {
      case TimeRangeComparisonEnum.DayBefore:
        {
          return {
            value: dayAndWeekBeforeToOffset[TimeRangeComparisonEnum.DayBefore],
            text: _i18n.i18n.translate('xpack.apm.timeComparison.select.dayBefore', {
              defaultMessage: 'Day before'
            })
          };
        }
      case TimeRangeComparisonEnum.WeekBefore:
        {
          return {
            value: dayAndWeekBeforeToOffset[TimeRangeComparisonEnum.WeekBefore],
            text: _i18n.i18n.translate('xpack.apm.timeComparison.select.weekBefore', {
              defaultMessage: 'Week before'
            })
          };
        }
      case TimeRangeComparisonEnum.PeriodBefore:
        {
          const offset = `${msDiff}ms`;
          const {
            startWithOffset,
            endWithOffset
          } = (0, _get_offset_in_ms.getOffsetInMs)({
            start: start.valueOf(),
            end: end.valueOf(),
            offset
          });
          return {
            value: offset,
            text: formatDate({
              currentPeriodEnd: end,
              previousPeriodStart: (0, _moment.default)(startWithOffset),
              previousPeriodEnd: (0, _moment.default)(endWithOffset)
            })
          };
        }
    }
  }).filter(isDefined);
}
function getComparisonOptions({
  start,
  end,
  showSelectedBoundsOption,
  anomalyDetectionJobsStatus,
  anomalyDetectionJobsData,
  preferredEnvironment
}) {
  const momentStart = (0, _moment.default)(start);
  const momentEnd = (0, _moment.default)(end);
  const msDiff = momentEnd.diff(momentStart, 'ms', true);
  let comparisonTypes;
  if (msDiff < twentyFiveHoursInMs) {
    // Less than 25 hours. This is because relative times may be rounded when
    // asking for a day, which can result in a duration > 24h. (e.g. rangeFrom: 'now-24h/h, rangeTo: 'now')
    comparisonTypes = [TimeRangeComparisonEnum.DayBefore, TimeRangeComparisonEnum.WeekBefore];
  } else if (msDiff < eightDaysInMs) {
    // Less than 8 days. This is because relative times may be rounded when
    // asking for a week, which can result in a duration > 7d. (e.g. rangeFrom: 'now-7d/d, rangeTo: 'now')
    comparisonTypes = [TimeRangeComparisonEnum.WeekBefore];
  } else {
    comparisonTypes = [TimeRangeComparisonEnum.PeriodBefore];
  }
  const hasMLJob = isDefined(anomalyDetectionJobsData) && anomalyDetectionJobsData.jobs.length > 0;
  const comparisonOptions = getSelectOptions({
    comparisonTypes,
    start: momentStart,
    end: momentEnd,
    msDiff
  });
  if (showSelectedBoundsOption && hasMLJob) {
    const disabled = anomalyDetectionJobsStatus === 'success' && !anomalyDetectionJobsData.jobs.some(j => j.environment === preferredEnvironment);
    comparisonOptions.push({
      value: TimeRangeComparisonEnum.ExpectedBounds,
      text: disabled ? _i18n.i18n.translate('xpack.apm.comparison.mlExpectedBoundsDisabledText', {
        defaultMessage: 'Expected bounds (Anomaly detection must be enabled for env)'
      }) : _i18n.i18n.translate('xpack.apm.comparison.mlExpectedBoundsText', {
        defaultMessage: 'Expected bounds'
      }),
      disabled
    });
  }
  return comparisonOptions;
}