"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateMaintenanceWindowRoute = void 0;
var _lib = require("../../../../lib");
var _types = require("../../../../../types");
var _common = require("../../../../../../common");
var _update = require("../../../../../../common/routes/maintenance_window/external/apis/update");
var _response = require("../../../../../../common/routes/maintenance_window/external/response");
var _transforms = require("../common/transforms");
var _schedule = require("../../../../../../common/routes/schedule");
var _transform_update_body = require("./transform_update_body");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const updateMaintenanceWindowRoute = (router, licenseState) => {
  router.patch({
    path: _types.UPDATE_MAINTENANCE_WINDOW_API_PATH,
    validate: {
      request: {
        body: _update.updateMaintenanceWindowRequestBodySchemaV1,
        params: _update.updateMaintenanceWindowRequestParamsSchemaV1
      },
      response: {
        200: {
          body: () => _response.maintenanceWindowResponseSchemaV1,
          description: 'Indicates a successful call.'
        },
        400: {
          description: 'Indicates an invalid schema or parameters.'
        },
        403: {
          description: 'Indicates that this call is forbidden.'
        },
        404: {
          description: 'Indicates a maintenance window with the given ID does not exist.'
        },
        409: {
          description: 'Indicates that the maintenance window has already been updated by another user.'
        }
      }
    },
    security: {
      authz: {
        requiredPrivileges: [`${_common.MAINTENANCE_WINDOW_API_PRIVILEGES.WRITE_MAINTENANCE_WINDOW}`]
      }
    },
    options: {
      access: 'public',
      summary: 'Update a maintenance window.',
      tags: ['oas-tag:maintenance-window'],
      availability: {
        since: '8.19.0',
        stability: 'stable'
      }
    }
  }, router.handleLegacyErrors((0, _lib.verifyAccessAndContext)(licenseState, async function (context, req, res) {
    var _body$schedule;
    licenseState.ensureLicenseForMaintenanceWindow();
    const body = req.body;
    const params = req.params;
    const customSchedule = body === null || body === void 0 ? void 0 : (_body$schedule = body.schedule) === null || _body$schedule === void 0 ? void 0 : _body$schedule.custom;
    const maintenanceWindowClient = (await context.alerting).getMaintenanceWindowClient();
    const maintenanceWindow = await maintenanceWindowClient.update({
      id: params.id,
      data: (0, _transform_update_body.transformUpdateBodyV1)(body)
    });
    const response = (0, _transforms.transformInternalMaintenanceWindowToExternalV1)(maintenanceWindow);

    // Return request duration in response when both are same otherwise throw an error
    const requestDurationInMilliseconds = customSchedule !== null && customSchedule !== void 0 && customSchedule.duration ? (0, _schedule.getDurationInMilliseconds)(customSchedule.duration) : undefined;
    const responseDurationInMilliseconds = (0, _schedule.getDurationInMilliseconds)(response.schedule.custom.duration);
    if (requestDurationInMilliseconds && requestDurationInMilliseconds !== responseDurationInMilliseconds) {
      throw new Error('Request duration does not match response duration.');
    }
    const responseDuration = customSchedule !== null && customSchedule !== void 0 && customSchedule.duration ? customSchedule.duration : response.schedule.custom.duration;
    return res.ok({
      body: {
        ...response,
        schedule: {
          custom: {
            ...response.schedule.custom,
            ...{
              duration: responseDuration
            }
          }
        }
      }
    });
  })));
};
exports.updateMaintenanceWindowRoute = updateMaintenanceWindowRoute;