"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ShippersRegistry = void 0;
var ShippersRegistry = /** @class */ (function () {
    function ShippersRegistry() {
        /**
         * Holds all the shippers: global and eventTypeExclusive.
         * This helps to avoid looping over all the shippers when we just need them all.
         */
        this.allShippers = new Map();
        /**
         * Holds the shippers that are not registered as exclusive to any event-type
         */
        this.globalShippers = new Map();
        /**
         * Holds the shippers that are exclusive to an event-type in the format of { [eventType]: ShippersMap }
         */
        this.eventTypeExclusiveShippers = new Map();
    }
    /**
     * Adds shipper to the registry.
     * @param shipperName The unique name of the shipper.
     * @param shipper The initialized shipper.
     */
    ShippersRegistry.prototype.addGlobalShipper = function (shipperName, shipper) {
        if (this.globalShippers.get(shipperName)) {
            throw new Error("Shipper \"".concat(shipperName, "\" is already registered"));
        }
        this.globalShippers.set(shipperName, shipper);
        this.allShippers.set(shipperName, shipper);
    };
    /**
     * Adds an event-type exclusive shipper.
     * @param eventType The name of the event type
     * @param shipperName The unique name for the shipper.
     * @param shipper The initialized shipper.
     */
    ShippersRegistry.prototype.addEventExclusiveShipper = function (eventType, shipperName, shipper) {
        var eventExclusiveMap = this.eventTypeExclusiveShippers.get(eventType) || new Map();
        if (eventExclusiveMap.get(shipperName)) {
            throw new Error("".concat(shipperName, " is already registered for event-type ").concat(eventType));
        }
        eventExclusiveMap.set(shipperName, shipper);
        this.eventTypeExclusiveShippers.set(eventType, eventExclusiveMap);
        this.allShippers.set(shipperName, shipper);
    };
    /**
     * Returns the shippers that must be used for the specified event type.
     * @param eventType The name of the event type.
     */
    ShippersRegistry.prototype.getShippersForEventType = function (eventType) {
        return this.eventTypeExclusiveShippers.get(eventType) || this.globalShippers;
    };
    return ShippersRegistry;
}());
exports.ShippersRegistry = ShippersRegistry;
