"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DefaultAlertService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/alerting-plugin/server");
var _synthetics_settings = require("../../saved_objects/synthetics_settings");
var _alert_actions = require("../../../common/rules/alert_actions");
var _translations = require("../../../common/rules/synthetics/translations");
var _synthetics_alerts = require("../../../common/constants/synthetics_alerts");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class DefaultAlertService {
  constructor(context, server, soClient) {
    (0, _defineProperty2.default)(this, "context", void 0);
    (0, _defineProperty2.default)(this, "soClient", void 0);
    (0, _defineProperty2.default)(this, "server", void 0);
    (0, _defineProperty2.default)(this, "settings", void 0);
    this.context = context;
    this.server = server;
    this.soClient = soClient;
  }
  async getSettings() {
    if (!this.settings) {
      this.settings = await (0, _synthetics_settings.getSyntheticsDynamicSettings)(this.soClient);
    }
    return this.settings;
  }
  async setupDefaultAlerts() {
    this.settings = await this.getSettings();
    const [statusRule, tlsRule] = await Promise.allSettled([this.setupStatusRule(), this.setupTlsRule()]);
    if (statusRule.status === 'rejected') {
      throw statusRule.reason;
    }
    if (tlsRule.status === 'rejected') {
      throw tlsRule.reason;
    }
    return {
      statusRule: statusRule.status === 'fulfilled' && statusRule.value ? statusRule.value : null,
      tlsRule: tlsRule.status === 'fulfilled' && tlsRule.value ? tlsRule.value : null
    };
  }
  getMinimumRuleInterval() {
    const minimumInterval = this.server.alerting.getConfig().minimumScheduleInterval;
    const minimumIntervalInMs = (0, _server.parseDuration)(minimumInterval.value);
    const defaultIntervalInMs = (0, _server.parseDuration)('1m');
    const interval = minimumIntervalInMs < defaultIntervalInMs ? '1m' : minimumInterval.value;
    return interval;
  }
  setupStatusRule() {
    var _this$settings;
    const minimumRuleInterval = this.getMinimumRuleInterval();
    if (((_this$settings = this.settings) === null || _this$settings === void 0 ? void 0 : _this$settings.defaultStatusRuleEnabled) === false) {
      return;
    }
    return this.createDefaultRuleIfNotExist(_synthetics_alerts.SYNTHETICS_STATUS_RULE, `Synthetics status internal rule`, minimumRuleInterval);
  }
  setupTlsRule() {
    var _this$settings2;
    const minimumRuleInterval = this.getMinimumRuleInterval();
    if (((_this$settings2 = this.settings) === null || _this$settings2 === void 0 ? void 0 : _this$settings2.defaultTLSRuleEnabled) === false) {
      return;
    }
    return this.createDefaultRuleIfNotExist(_synthetics_alerts.SYNTHETICS_TLS_RULE, `Synthetics internal TLS rule`, minimumRuleInterval);
  }
  async getExistingAlert(ruleType) {
    var _await$this$context$a;
    const rulesClient = await ((_await$this$context$a = await this.context.alerting) === null || _await$this$context$a === void 0 ? void 0 : _await$this$context$a.getRulesClient());
    const {
      data
    } = await rulesClient.find({
      options: {
        page: 1,
        perPage: 1,
        filter: `alert.attributes.alertTypeId:(${ruleType}) AND alert.attributes.tags:"SYNTHETICS_DEFAULT_ALERT"`
      }
    });
    if (data.length === 0) {
      return;
    }
    const {
      actions = [],
      systemActions = [],
      ...alert
    } = data[0];
    return {
      ...alert,
      actions: [...actions, ...systemActions],
      ruleTypeId: alert.alertTypeId
    };
  }
  async createDefaultRuleIfNotExist(ruleType, name, interval) {
    var _await$this$context$a2;
    const alert = await this.getExistingAlert(ruleType);
    if (alert) {
      return alert;
    }
    const actions = await this.getAlertActions(ruleType);
    const rulesClient = await ((_await$this$context$a2 = await this.context.alerting) === null || _await$this$context$a2 === void 0 ? void 0 : _await$this$context$a2.getRulesClient());
    const {
      actions: actionsFromRules = [],
      systemActions = [],
      ...newAlert
    } = await rulesClient.create({
      data: {
        actions,
        params: {},
        consumer: 'uptime',
        alertTypeId: ruleType,
        schedule: {
          interval
        },
        tags: ['SYNTHETICS_DEFAULT_ALERT'],
        name,
        enabled: true,
        throttle: null
      }
    });
    return {
      ...newAlert,
      actions: [...actionsFromRules, ...systemActions],
      ruleTypeId: newAlert.alertTypeId
    };
  }
  async updateStatusRule(enabled) {
    const minimumRuleInterval = this.getMinimumRuleInterval();
    if (enabled) {
      return this.upsertDefaultAlert(_synthetics_alerts.SYNTHETICS_STATUS_RULE, `Synthetics status internal rule`, minimumRuleInterval);
    } else {
      var _await$this$context$a3;
      const rulesClient = await ((_await$this$context$a3 = await this.context.alerting) === null || _await$this$context$a3 === void 0 ? void 0 : _await$this$context$a3.getRulesClient());
      await rulesClient.bulkDeleteRules({
        filter: `alert.attributes.alertTypeId:"${_synthetics_alerts.SYNTHETICS_STATUS_RULE}" AND alert.attributes.tags:"SYNTHETICS_DEFAULT_ALERT"`
      });
    }
  }
  async updateTlsRule(enabled) {
    const minimumRuleInterval = this.getMinimumRuleInterval();
    if (enabled) {
      return this.upsertDefaultAlert(_synthetics_alerts.SYNTHETICS_TLS_RULE, `Synthetics internal TLS rule`, minimumRuleInterval);
    } else {
      var _await$this$context$a4;
      const rulesClient = await ((_await$this$context$a4 = await this.context.alerting) === null || _await$this$context$a4 === void 0 ? void 0 : _await$this$context$a4.getRulesClient());
      await rulesClient.bulkDeleteRules({
        filter: `alert.attributes.alertTypeId:"${_synthetics_alerts.SYNTHETICS_TLS_RULE}" AND alert.attributes.tags:"SYNTHETICS_DEFAULT_ALERT"`
      });
    }
  }
  async upsertDefaultAlert(ruleType, name, interval) {
    var _await$this$context$a5;
    const rulesClient = await ((_await$this$context$a5 = await this.context.alerting) === null || _await$this$context$a5 === void 0 ? void 0 : _await$this$context$a5.getRulesClient());
    const alert = await this.getExistingAlert(ruleType);
    if (alert) {
      const currentIntervalInMs = (0, _server.parseDuration)(alert.schedule.interval);
      const minimumIntervalInMs = (0, _server.parseDuration)(interval);
      const actions = await this.getAlertActions(ruleType);
      const {
        actions: actionsFromRules = [],
        systemActions = [],
        ...updatedAlert
      } = await rulesClient.update({
        id: alert.id,
        data: {
          actions,
          name: alert.name,
          tags: alert.tags,
          schedule: {
            interval: currentIntervalInMs < minimumIntervalInMs ? interval : alert.schedule.interval
          },
          params: alert.params
        }
      });
      return {
        ...updatedAlert,
        actions: [...actionsFromRules, ...systemActions],
        ruleTypeId: updatedAlert.alertTypeId
      };
    }
    return await this.createDefaultRuleIfNotExist(ruleType, name, interval);
  }
  async getAlertActions(ruleType) {
    const {
      actionConnectors,
      settings
    } = await this.getActionConnectors();
    const defaultActions = (actionConnectors !== null && actionConnectors !== void 0 ? actionConnectors : []).filter(act => {
      var _settings$defaultConn;
      return settings === null || settings === void 0 ? void 0 : (_settings$defaultConn = settings.defaultConnectors) === null || _settings$defaultConn === void 0 ? void 0 : _settings$defaultConn.includes(act.id);
    });
    if (ruleType === _synthetics_alerts.SYNTHETICS_STATUS_RULE) {
      return (0, _alert_actions.populateAlertActions)({
        defaultActions,
        groupId: _synthetics_alerts.ACTION_GROUP_DEFINITIONS.MONITOR_STATUS.id,
        defaultEmail: settings === null || settings === void 0 ? void 0 : settings.defaultEmail,
        translations: {
          defaultActionMessage: _translations.SyntheticsMonitorStatusTranslations.defaultActionMessage,
          defaultRecoveryMessage: _translations.SyntheticsMonitorStatusTranslations.defaultRecoveryMessage,
          defaultSubjectMessage: _translations.SyntheticsMonitorStatusTranslations.defaultSubjectMessage,
          defaultRecoverySubjectMessage: _translations.SyntheticsMonitorStatusTranslations.defaultRecoverySubjectMessage
        }
      });
    } else {
      return (0, _alert_actions.populateAlertActions)({
        defaultActions,
        groupId: _synthetics_alerts.ACTION_GROUP_DEFINITIONS.TLS_CERTIFICATE.id,
        defaultEmail: settings === null || settings === void 0 ? void 0 : settings.defaultEmail,
        translations: {
          defaultActionMessage: _translations.TlsTranslations.defaultActionMessage,
          defaultRecoveryMessage: _translations.TlsTranslations.defaultRecoveryMessage,
          defaultSubjectMessage: _translations.TlsTranslations.defaultSubjectMessage,
          defaultRecoverySubjectMessage: _translations.TlsTranslations.defaultRecoverySubjectMessage
        }
      });
    }
  }
  async getActionConnectors() {
    var _await$this$context$a6;
    const actionsClient = (_await$this$context$a6 = await this.context.actions) === null || _await$this$context$a6 === void 0 ? void 0 : _await$this$context$a6.getActionsClient();
    if (!this.settings) {
      this.settings = await (0, _synthetics_settings.getSyntheticsDynamicSettings)(this.soClient);
    }
    let actionConnectors = [];
    try {
      actionConnectors = await actionsClient.getAll();
    } catch (error) {
      this.server.logger.error(`Error getting connectors, Error: ${error.message}`, {
        error
      });
    }
    return {
      actionConnectors,
      settings: this.settings
    };
  }
}
exports.DefaultAlertService = DefaultAlertService;