"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.resyncStreamsRoute = exports.managementRoutes = exports.getStreamsStatusRoute = exports.forkStreamsRoute = void 0;
var _streamsSchema = require("@kbn/streams-schema");
var _zod = require("@kbn/zod");
var _constants = require("../../../../common/constants");
var _create_server_route = require("../../create_server_route");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const forkStreamsRoute = exports.forkStreamsRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'POST /api/streams/{name}/_fork 2023-10-31',
  options: {
    access: 'public',
    description: 'Forks a wired stream and creates a child stream',
    summary: 'Fork a stream',
    availability: {
      stability: 'experimental'
    }
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.manage]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    }),
    body: _zod.z.object({
      stream: _zod.z.object({
        name: _zod.z.string()
      }),
      if: _streamsSchema.conditionSchema
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients
  }) => {
    const {
      streamsClient
    } = await getScopedClients({
      request
    });
    return await streamsClient.forkStream({
      parent: params.path.name,
      if: params.body.if,
      name: params.body.stream.name
    });
  }
});
const resyncStreamsRoute = exports.resyncStreamsRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'POST /api/streams/_resync 2023-10-31',
  options: {
    access: 'public',
    description: 'Resyncs all streams, making sure that Elasticsearch assets are up to date',
    summary: 'Resync streams',
    availability: {
      stability: 'experimental'
    }
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.manage]
    }
  },
  params: _zod.z.object({}),
  handler: async ({
    request,
    getScopedClients
  }) => {
    const {
      streamsClient
    } = await getScopedClients({
      request
    });
    return await streamsClient.resyncStreams();
  }
});
const getStreamsStatusRoute = exports.getStreamsStatusRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'GET /api/streams/_status',
  options: {
    access: 'internal'
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  handler: async ({
    request,
    getScopedClients
  }) => {
    const {
      streamsClient
    } = await getScopedClients({
      request
    });
    return {
      enabled: await streamsClient.isStreamsEnabled()
    };
  }
});
const managementRoutes = exports.managementRoutes = {
  ...forkStreamsRoute,
  ...resyncStreamsRoute,
  ...getStreamsStatusRoute
};