"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.transformDatatableToEsqlTable = exports.toGroupedEsqlQueryHits = exports.toEsqlQueryHits = exports.rowToDocument = exports.getEsqlQueryHits = exports.getAlertIdFields = exports.ALERT_ID_SUGGESTED_MAX = exports.ALERT_ID_COLUMN = void 0;
var _lodash = require("lodash");
var _esqlAst = require("@kbn/esql-ast");
var _esqlValidationAutocomplete = require("@kbn/esql-validation-autocomplete");
var _esqlUtils = require("@kbn/esql-utils");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ESQL_DOCUMENT_ID = 'esql_query_document';
const ALERT_ID_COLUMN = exports.ALERT_ID_COLUMN = 'Alert ID';
const ALERT_ID_SUGGESTED_MAX = exports.ALERT_ID_SUGGESTED_MAX = 10;
const rowToDocument = (columns, row) => {
  return columns.reduce((acc, column, i) => {
    acc[column.name] = row[i];
    return acc;
  }, {});
};
exports.rowToDocument = rowToDocument;
const getEsqlQueryHits = (table, query, isGroupAgg) => {
  if (isGroupAgg) {
    const alertIdFields = getAlertIdFields(query, table.columns);
    return toGroupedEsqlQueryHits(table, alertIdFields);
  }
  return toEsqlQueryHits(table);
};
exports.getEsqlQueryHits = getEsqlQueryHits;
const toEsqlQueryHits = table => {
  const hits = [];
  const rows = [];
  for (const row of table.values) {
    const document = rowToDocument(table.columns, row);
    hits.push({
      _id: ESQL_DOCUMENT_ID,
      _index: '',
      _source: document
    });
    rows.push(rows.length > 0 ? document : {
      [ALERT_ID_COLUMN]: _constants.ActionGroupId,
      ...document
    });
  }
  return {
    results: {
      isCountAgg: true,
      isGroupAgg: false,
      esResult: {
        took: 0,
        timed_out: false,
        _shards: {
          failed: 0,
          successful: 0,
          total: 0
        },
        hits: {
          hits,
          total: hits.length
        }
      }
    },
    rows,
    cols: getColumnsForPreview(table.columns)
  };
};
exports.toEsqlQueryHits = toEsqlQueryHits;
const toGroupedEsqlQueryHits = (table, alertIdFields) => {
  const duplicateAlertIds = new Set();
  const longAlertIds = new Set();
  const rows = [];
  const mappedAlertIds = {};
  const groupedHits = table.values.reduce((acc, row) => {
    const document = rowToDocument(table.columns, row);
    const mappedAlertId = alertIdFields.filter(a => !(0, _lodash.isNil)(document[a])).map(a => document[a]);
    if (mappedAlertId.length > 0) {
      const alertId = mappedAlertId.join(',');
      const hit = {
        _id: ESQL_DOCUMENT_ID,
        _index: '',
        _source: document
      };
      if (acc[alertId]) {
        duplicateAlertIds.add(alertId);
        acc[alertId].push(hit);
      } else {
        acc[alertId] = [hit];
        mappedAlertIds[alertId] = mappedAlertId;
      }
      rows.push({
        [ALERT_ID_COLUMN]: alertId,
        ...document
      });
      if (mappedAlertId.length >= ALERT_ID_SUGGESTED_MAX) {
        longAlertIds.add(alertId);
      }
    }
    return acc;
  }, {});
  const aggregations = {
    groupAgg: {
      buckets: (0, _lodash.entries)(groupedHits).map(([key, value]) => {
        return {
          key: mappedAlertIds[key],
          doc_count: value.length,
          topHitsAgg: {
            hits: {
              hits: value
            }
          }
        };
      })
    }
  };
  return {
    results: {
      isCountAgg: false,
      isGroupAgg: true,
      esResult: {
        took: 0,
        timed_out: false,
        _shards: {
          failed: 0,
          successful: 0,
          total: 0
        },
        hits: {
          hits: []
        },
        aggregations
      },
      termField: alertIdFields
    },
    duplicateAlertIds,
    longAlertIds,
    rows,
    cols: getColumnsForPreview(table.columns)
  };
};
exports.toGroupedEsqlQueryHits = toGroupedEsqlQueryHits;
const transformDatatableToEsqlTable = datatable => {
  const columns = datatable.columns.map(c => ({
    name: c.id,
    type: c.meta.type
  }));
  const values = datatable.rows.map(r => Object.values(r));
  return {
    columns,
    values
  };
};
exports.transformDatatableToEsqlTable = transformDatatableToEsqlTable;
const getAlertIdFields = (query, resultColumns) => {
  const {
    root
  } = (0, _esqlAst.parse)(query);
  const commands = root.commands;
  const columns = resultColumns.map(c => c.name);

  // Check for STATS command
  const statsCommandIndex = getLastStatsCommandIndex(commands);
  if (statsCommandIndex > -1) {
    const statsCommand = commands[statsCommandIndex];
    // Check for BY option and get fields
    const byOption = getByOption(statsCommand);
    if (byOption) {
      let fields = getFields(byOption);

      // Check if any STATS fields were renamed
      const renameCommands = getRenameCommands(commands.slice(statsCommandIndex));
      if (renameCommands.length > 0) {
        fields = getFieldsFromRenameCommands(renameCommands, fields);
      }

      // Check if any fields were dropped
      fields = (0, _lodash.intersection)(fields, columns);
      if (fields.length > 0) {
        return fields;
      }
    }
  }

  // Check for METADATA _id
  const metadataOption = getMetadataOption(commands);
  if (metadataOption) {
    const fields = getIdField(metadataOption);

    // Check if _id was dropped
    if ((0, _lodash.intersection)(fields, columns).length > 0) {
      return fields;
    }
  }

  // Return all columns
  return columns;
};
exports.getAlertIdFields = getAlertIdFields;
const getLastStatsCommandIndex = commands => (0, _lodash.findLastIndex)(commands, c => c.name === 'stats');
const getByOption = astCommand => {
  for (const statsArg of astCommand.args) {
    if ((0, _esqlValidationAutocomplete.isOptionItem)(statsArg) && statsArg.name === 'by') {
      return statsArg;
    }
  }
};
const getFields = option => {
  const fields = [];
  for (const arg of option.args) {
    if ((0, _esqlValidationAutocomplete.isColumnItem)(arg)) {
      fields.push(arg.name);
    }
  }
  return fields;
};
const getRenameCommands = commands => commands.filter(c => c.name === 'rename');
const getFieldsFromRenameCommands = (astCommands, fields) => {
  return astCommands.reduce((updatedFields, command) => {
    for (const renameArg of command.args) {
      if ((0, _esqlValidationAutocomplete.isFunctionItem)(renameArg)) {
        const {
          original,
          renamed
        } = (0, _esqlUtils.getArgsFromRenameFunction)(renameArg);
        if ((0, _esqlValidationAutocomplete.isColumnItem)(original) && (0, _esqlValidationAutocomplete.isColumnItem)(renamed)) {
          updatedFields = updatedFields.map(field => field === original.name ? renamed.name : field);
        }
      }
    }
    return updatedFields;
  }, fields);
};
const getMetadataOption = commands => {
  const fromCommand = commands.find(c => c.name === 'from');
  if (fromCommand) {
    for (const fromArg of fromCommand.args) {
      if ((0, _esqlValidationAutocomplete.isOptionItem)(fromArg) && fromArg.name === 'metadata') {
        return fromArg;
      }
    }
  }
  return undefined;
};
const getIdField = option => {
  const fields = [];
  for (const arg of option.args) {
    if ((0, _esqlValidationAutocomplete.isColumnItem)(arg) && arg.name === '_id') {
      fields.push(arg.name);
      return fields;
    }
  }
  return fields;
};
const getColumnsForPreview = columns => {
  const cols = [{
    id: ALERT_ID_COLUMN,
    actions: false
  }];
  for (const c of columns) {
    cols.push({
      id: c.name,
      actions: false
    });
  }
  return cols;
};