"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initPutSpacesApi = initPutSpacesApi;
var _configSchema = require("@kbn/config-schema");
var _server = require("@kbn/core/server");
var _common = require("../../../../common");
var _errors = require("../../../lib/errors");
var _space_schema = require("../../../lib/space_schema");
var _lib = require("../../lib");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function initPutSpacesApi(deps) {
  const {
    router,
    getSpacesService,
    isServerless
  } = deps;
  router.versioned.put({
    path: '/api/spaces/space/{id}',
    access: 'public',
    summary: `Update a space`,
    options: {
      tags: ['oas-tag:spaces']
    }
  }).addVersion({
    version: _common.API_VERSIONS.public.v1,
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the spaces service via a scoped spaces client'
      }
    },
    validate: {
      request: {
        params: _configSchema.schema.object({
          id: _configSchema.schema.string({
            meta: {
              description: 'The space identifier. You are unable to change the ID with the update operation.'
            }
          })
        }),
        body: (0, _space_schema.getSpaceSchema)(isServerless)
      },
      response: {
        200: {
          description: 'Indicates a successful call.'
        }
      }
    }
  }, (0, _lib.createLicensedRouteHandler)(async (context, request, response) => {
    const spacesClient = getSpacesService().createSpacesClient(request);
    const space = request.body;
    const id = request.params.id;
    let result;
    try {
      result = await spacesClient.update(id, {
        ...space
      });
    } catch (error) {
      if (_server.SavedObjectsErrorHelpers.isNotFoundError(error)) {
        return response.notFound();
      }
      return response.customError((0, _errors.wrapError)(error));
    }
    return response.ok({
      body: result
    });
  }));
}