"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildParamValues = exports.SyntheticsAvailabilityTransformGenerator = void 0;
var _sloSchema = require("@kbn/slo-schema");
var _ = require(".");
var _constants = require("../../../common/constants");
var _slo_transform_template = require("../../assets/transform_templates/slo_transform_template");
var _errors = require("../../errors");
var _common = require("./common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class SyntheticsAvailabilityTransformGenerator extends _.TransformGenerator {
  constructor(spaceId, dataViewService, isServerless) {
    super(spaceId, dataViewService, isServerless);
  }
  async getTransformParams(slo) {
    if (!_sloSchema.syntheticsAvailabilityIndicatorSchema.is(slo.indicator)) {
      throw new _errors.InvalidTransformError(`Cannot handle SLO of indicator type: ${slo.indicator.type}`);
    }
    return (0, _slo_transform_template.getSLOTransformTemplate)(this.buildTransformId(slo), this.buildDescription(slo), await this.buildSource(slo, slo.indicator), this.buildDestination(slo), this.buildGroupBy(slo, slo.indicator), this.buildAggregations(slo), this.buildSettings(slo, this.isServerless ? '@timestamp' : 'event.ingested'), slo);
  }
  buildTransformId(slo) {
    return (0, _constants.getSLOTransformId)(slo.id, slo.revision);
  }
  buildGroupBy(slo, indicator) {
    var _indicator$params$tag, _indicator$params$tag2, _indicator$params$pro, _indicator$params$pro2, _indicator$params$mon, _indicator$params$mon2;
    // These are the group by fields that will be used in `groupings` key
    // in the summary and rollup documents. For Synthetics, we want to use the
    // user-readable `monitor.name` and `observer.geo.name` fields by default,
    // unless otherwise specified by the user.
    const flattenedGroupBy = [slo.groupBy].flat().filter(value => !!value);
    const groupings = flattenedGroupBy.length && !flattenedGroupBy.includes(_sloSchema.ALL_VALUE) ? slo.groupBy : _constants.SYNTHETICS_DEFAULT_GROUPINGS;
    const hasTags = !((_indicator$params$tag = indicator.params.tags) !== null && _indicator$params$tag !== void 0 && _indicator$params$tag.find(param => param.value === _sloSchema.ALL_VALUE)) && ((_indicator$params$tag2 = indicator.params.tags) === null || _indicator$params$tag2 === void 0 ? void 0 : _indicator$params$tag2.length);
    const hasProjects = !((_indicator$params$pro = indicator.params.projects) !== null && _indicator$params$pro !== void 0 && _indicator$params$pro.find(param => param.value === _sloSchema.ALL_VALUE)) && ((_indicator$params$pro2 = indicator.params.projects) === null || _indicator$params$pro2 === void 0 ? void 0 : _indicator$params$pro2.length);
    const hasMonitorIds = !((_indicator$params$mon = indicator.params.monitorIds) !== null && _indicator$params$mon !== void 0 && _indicator$params$mon.find(param => param.value === _sloSchema.ALL_VALUE)) && ((_indicator$params$mon2 = indicator.params.monitorIds) === null || _indicator$params$mon2 === void 0 ? void 0 : _indicator$params$mon2.length);
    const includesDefaultGroupings = groupings.includes('monitor.name') && groupings.includes('observer.geo.name');
    // These groupBy fields must match the fields from the source query, otherwise
    // the transform will create permutations for each value present in the source.
    // E.g. if environment is not specified in the source query, but we include it in the groupBy,
    // we'll output documents for each environment value
    const extraGroupByFields = {
      /* additional fields needed to hyperlink back to the Synthetics app when
       * grouping by monitor.name and observer.geo.name.
       * `monitor.name` and `observer.geo.name` are the labels, while
       * observer.name and config_id are the values. We need the values
       * to build a URL back to Synthetics */
      ...(includesDefaultGroupings && {
        'observer.name': {
          terms: {
            field: 'observer.name'
          }
        },
        'observer.geo.name': {
          terms: {
            field: 'observer.geo.name'
          }
        },
        'monitor.config_id': {
          terms: {
            field: 'config_id'
          }
        },
        'monitor.name': {
          terms: {
            field: 'monitor.name'
          }
        }
      }),
      ...(hasMonitorIds && {
        'monitor.id': {
          terms: {
            field: 'monitor.id'
          }
        }
      }),
      ...(hasTags && {
        tags: {
          terms: {
            field: 'tags'
          }
        }
      }),
      ...(hasProjects && {
        'monitor.project.id': {
          terms: {
            field: 'monitor.project.id'
          }
        }
      })
    };
    return this.buildCommonGroupBy({
      ...slo,
      groupBy: groupings
    }, '@timestamp', extraGroupByFields);
  }
  async buildSource(slo, indicator) {
    const queryFilter = [{
      term: {
        'summary.final_attempt': true
      }
    }, {
      term: {
        'meta.space_id': this.spaceId
      }
    }, (0, _common.getFilterRange)(slo, '@timestamp', this.isServerless)];
    const {
      monitorIds,
      tags,
      projects
    } = buildParamValues({
      monitorIds: indicator.params.monitorIds || [],
      tags: indicator.params.tags || [],
      projects: indicator.params.projects || []
    });
    if (!monitorIds.includes(_sloSchema.ALL_VALUE) && monitorIds.length) {
      queryFilter.push({
        terms: {
          'monitor.id': monitorIds
        }
      });
    }
    if (!tags.includes(_sloSchema.ALL_VALUE) && tags.length) {
      queryFilter.push({
        terms: {
          tags
        }
      });
    }
    if (!projects.includes(_sloSchema.ALL_VALUE) && projects.length) {
      queryFilter.push({
        terms: {
          'monitor.project.id': projects
        }
      });
    }
    if (!!indicator.params.filter) {
      queryFilter.push((0, _.getElasticsearchQueryOrThrow)(indicator.params.filter));
    }
    const dataView = await this.getIndicatorDataView(indicator.params.dataViewId);
    return {
      index: _constants.SYNTHETICS_INDEX_PATTERN,
      runtime_mappings: this.buildCommonRuntimeMappings(dataView),
      query: {
        bool: {
          filter: queryFilter
        }
      }
    };
  }
  buildDestination(slo) {
    return {
      pipeline: (0, _constants.getSLOPipelineId)(slo.id, slo.revision),
      index: _constants.SLI_DESTINATION_INDEX_NAME
    };
  }
  buildAggregations(slo) {
    if (!_sloSchema.occurrencesBudgetingMethodSchema.is(slo.budgetingMethod)) {
      throw new Error("The sli.synthetics.availability indicator MUST have an 'Occurrences' budgeting method.");
    }
    return {
      'slo.numerator': {
        filter: {
          term: {
            'monitor.status': 'up'
          }
        }
      },
      'slo.denominator': {
        filter: {
          term: {
            'summary.final_attempt': true
          }
        }
      }
    };
  }
}
exports.SyntheticsAvailabilityTransformGenerator = SyntheticsAvailabilityTransformGenerator;
const buildParamValues = params => {
  return Object.keys(params).reduce((acc, key) => {
    var _params$key;
    return {
      ...acc,
      [key]: (_params$key = params[key]) === null || _params$key === void 0 ? void 0 : _params$key.map(p => p.value)
    };
  }, {});
};
exports.buildParamValues = buildParamValues;