"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SiemMigrationTelemetryClient = void 0;
var _events = require("../../../telemetry/event_based/events");
var _event_meta = require("../../../telemetry/event_based/event_meta");
var _types = require("../../../telemetry/event_based/types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class SiemMigrationTelemetryClient {
  constructor(telemetry, logger, migrationId, modelName = '') {
    this.telemetry = telemetry;
    this.logger = logger;
    this.migrationId = migrationId;
    this.modelName = modelName;
  }
  reportEvent(eventTypeOpts, data) {
    try {
      this.telemetry.reportEvent(eventTypeOpts.eventType, data);
    } catch (e) {
      this.logger.error(`Error reporting event ${eventTypeOpts.eventType}: ${e.message}`);
    }
  }
  reportIntegrationsMatch({
    preFilterIntegrations,
    postFilterIntegration
  }) {
    this.reportEvent(_events.SIEM_MIGRATIONS_INTEGRATIONS_MATCH, {
      model: this.modelName,
      migrationId: this.migrationId,
      preFilterIntegrationNames: preFilterIntegrations.map(integration => integration.id) || [],
      preFilterIntegrationCount: preFilterIntegrations.length,
      postFilterIntegrationName: postFilterIntegration ? postFilterIntegration.id : '',
      postFilterIntegrationCount: postFilterIntegration ? 1 : 0,
      eventName: _event_meta.siemMigrationEventNames[_types.SiemMigrationsEventTypes.IntegrationsMatch]
    });
  }
  reportPrebuiltRulesMatch({
    preFilterRules,
    postFilterRule
  }) {
    this.reportEvent(_events.SIEM_MIGRATIONS_PREBUILT_RULES_MATCH, {
      model: this.modelName,
      migrationId: this.migrationId,
      preFilterRuleNames: preFilterRules.map(rule => rule.rule_id) || [],
      preFilterRuleCount: preFilterRules.length,
      postFilterRuleName: postFilterRule ? postFilterRule.rule_id : '',
      postFilterRuleCount: postFilterRule ? 1 : 0,
      eventName: _event_meta.siemMigrationEventNames[_types.SiemMigrationsEventTypes.PrebuiltRulesMatch]
    });
  }
  startSiemMigrationTask() {
    const startTime = Date.now();
    const stats = {
      completed: 0,
      failed: 0
    };
    return {
      startRuleTranslation: () => {
        const ruleStartTime = Date.now();
        return {
          success: migrationResult => {
            var _migrationResult$elas;
            stats.completed++;
            this.reportEvent(_events.SIEM_MIGRATIONS_RULE_TRANSLATION_SUCCESS, {
              migrationId: this.migrationId,
              translationResult: migrationResult.translation_result || '',
              duration: Date.now() - ruleStartTime,
              model: this.modelName,
              prebuiltMatch: (_migrationResult$elas = migrationResult.elastic_rule) !== null && _migrationResult$elas !== void 0 && _migrationResult$elas.prebuilt_rule_id ? true : false,
              eventName: _event_meta.siemMigrationEventNames[_types.SiemMigrationsEventTypes.TranslationSuccess]
            });
          },
          failure: error => {
            stats.failed++;
            this.reportEvent(_events.SIEM_MIGRATIONS_RULE_TRANSLATION_FAILURE, {
              migrationId: this.migrationId,
              error: error.message,
              model: this.modelName,
              eventName: _event_meta.siemMigrationEventNames[_types.SiemMigrationsEventTypes.TranslationFailure]
            });
          }
        };
      },
      success: () => {
        const duration = Date.now() - startTime;
        this.reportEvent(_events.SIEM_MIGRATIONS_MIGRATION_SUCCESS, {
          migrationId: this.migrationId,
          model: this.modelName || '',
          completed: stats.completed,
          failed: stats.failed,
          total: stats.completed + stats.failed,
          duration,
          eventName: _event_meta.siemMigrationEventNames[_types.SiemMigrationsEventTypes.MigrationSuccess]
        });
      },
      failure: error => {
        const duration = Date.now() - startTime;
        this.reportEvent(_events.SIEM_MIGRATIONS_MIGRATION_FAILURE, {
          migrationId: this.migrationId,
          model: this.modelName || '',
          completed: stats.completed,
          failed: stats.failed,
          total: stats.completed + stats.failed,
          duration,
          error: error.message,
          eventName: _event_meta.siemMigrationEventNames[_types.SiemMigrationsEventTypes.MigrationFailure]
        });
      },
      aborted: error => {
        const duration = Date.now() - startTime;
        this.reportEvent(_events.SIEM_MIGRATIONS_MIGRATION_ABORTED, {
          migrationId: this.migrationId,
          model: this.modelName || '',
          completed: stats.completed,
          failed: stats.failed,
          total: stats.completed + stats.failed,
          duration,
          reason: error.message,
          eventName: _event_meta.siemMigrationEventNames[_types.SiemMigrationsEventTypes.MigrationAborted]
        });
      }
    };
  }
}
exports.SiemMigrationTelemetryClient = SiemMigrationTelemetryClient;