"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RuleMigrationsDataIntegrationsClient = void 0;
var _rule_migrations_data_base_client = require("./rule_migrations_data_base_client");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/* The minimum score required for a integration to be considered correct, might need to change this later */
const MIN_SCORE = 40;
/* The number of integrations the RAG will return, sorted by score */
const RETURNED_INTEGRATIONS = 5;

/* BULK_MAX_SIZE defines the number to break down the bulk operations by.
 * The 500 number was chosen as a reasonable number to avoid large payloads. It can be adjusted if needed.
 */
class RuleMigrationsDataIntegrationsClient extends _rule_migrations_data_base_client.RuleMigrationsDataBaseClient {
  /** Returns the Security integration packages that have "logs" type `data_streams` configured, including pre-release packages */
  async getSecurityLogsPackages() {
    var _this$dependencies$pa;
    const packages = await ((_this$dependencies$pa = this.dependencies.packageService) === null || _this$dependencies$pa === void 0 ? void 0 : _this$dependencies$pa.asInternalUser.getPackages({
      prerelease: true,
      category: 'security'
    }));
    return packages === null || packages === void 0 ? void 0 : packages.filter(pkg => {
      var _pkg$data_streams;
      return (_pkg$data_streams = pkg.data_streams) === null || _pkg$data_streams === void 0 ? void 0 : _pkg$data_streams.some(({
        type
      }) => type === 'logs');
    });
  }

  /** Indexes an array of integrations to be used with ELSER semantic search queries */
  async populate() {
    const index = await this.getIndexName();
    const packages = await this.getSecurityLogsPackages();
    if (packages) {
      const ragIntegrations = packages.reduce((acc, pkg) => {
        var _pkg$data_streams2;
        const logsDataStreams = (_pkg$data_streams2 = pkg.data_streams) === null || _pkg$data_streams2 === void 0 ? void 0 : _pkg$data_streams2.filter(({
          type
        }) => type === 'logs');
        // Only include packages that have logs data streams
        if (logsDataStreams !== null && logsDataStreams !== void 0 && logsDataStreams.length) {
          acc.push({
            title: pkg.title,
            id: pkg.name,
            description: (pkg === null || pkg === void 0 ? void 0 : pkg.description) || '',
            data_streams: logsDataStreams.map(stream => ({
              dataset: stream.dataset,
              index_pattern: `${stream.type}-${stream.dataset}-*`,
              title: stream.title
            })),
            elser_embedding: [pkg.title, pkg.description, ...logsDataStreams.map(stream => stream.title)].join(' - ')
          });
        }
        return acc;
      }, []);
      if (ragIntegrations.length === 0) {
        this.logger.debug('No security integrations with logs data streams found to index');
        return;
      }
      await this.esClient.bulk({
        refresh: 'wait_for',
        operations: ragIntegrations.flatMap(({
          id,
          ...doc
        }) => [{
          update: {
            _index: index,
            _id: id
          }
        }, {
          doc,
          doc_as_upsert: true
        }])
      }, {
        requestTimeout: 10 * 60 * 1000
      } // 10 minutes
      ).then(response => {
        if (response.errors) {
          var _response$items$find, _response$items$find$, _response$items$find$2;
          // use the first error to throw
          const reason = (_response$items$find = response.items.find(item => {
            var _item$update;
            return (_item$update = item.update) === null || _item$update === void 0 ? void 0 : _item$update.error;
          })) === null || _response$items$find === void 0 ? void 0 : (_response$items$find$ = _response$items$find.update) === null || _response$items$find$ === void 0 ? void 0 : (_response$items$find$2 = _response$items$find$.error) === null || _response$items$find$2 === void 0 ? void 0 : _response$items$find$2.reason;
          throw new Error(reason !== null && reason !== void 0 ? reason : 'Unknown error');
        }
      }).catch(error => {
        this.logger.error(`Error indexing integrations embeddings: ${error.message}`);
        throw error;
      });
    } else {
      this.logger.warn('Package service not available, not able not populate integrations index');
    }
  }

  /** Retrieves the integration details for a given semantic query */
  async semanticSearch(semanticQuery) {
    const index = await this.getIndexName();
    const query = {
      bool: {
        should: [{
          semantic: {
            query: semanticQuery,
            field: 'elser_embedding',
            boost: 1.5
          }
        }, {
          multi_match: {
            query: semanticQuery,
            fields: ['title^2', 'description'],
            boost: 3
          }
        }],
        // Filter to ensure we only return integrations that have data streams
        // because there may be integrations without data streams indexed in old migrations
        filter: {
          exists: {
            field: 'data_streams'
          }
        }
      }
    };
    const results = await this.esClient.search({
      index,
      query,
      size: RETURNED_INTEGRATIONS,
      min_score: MIN_SCORE
    }).then(this.processResponseHits.bind(this)).catch(error => {
      this.logger.error(`Error querying integration details for ELSER: ${error.message}`);
      throw error;
    });
    return results;
  }
}
exports.RuleMigrationsDataIntegrationsClient = RuleMigrationsDataIntegrationsClient;