"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchIndexPatterns = void 0;
var _langgraph = require("@langchain/langgraph");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const fetchIndexPatterns = ({
  esClient
}) => {
  return async state => {
    try {
      const indicesResolveIndexResponse = await esClient.indices.resolveIndex({
        name: '*',
        expand_wildcards: 'open'
      }, {
        requestTimeout: 20000 // 20 seconds
      });

      // Stores indices that do not have any datastreams or aliases
      const indicesWithoutDatastreamsOrAliases = new Set();
      const seenIndices = new Set();
      const dataStreamsAndAliases = new Set();

      // Collect all indices from data streams and aliases
      for (const dataStream of indicesResolveIndexResponse.data_streams) {
        for (const index of dataStream.backing_indices) {
          seenIndices.add(index);
        }
        dataStreamsAndAliases.add(dataStream.name);
      }
      for (const alias of indicesResolveIndexResponse.aliases) {
        for (const index of alias.indices) {
          seenIndices.add(index);
        }
        dataStreamsAndAliases.add(alias.name);
      }

      // Add indices that do not have any datastreams or aliases
      for (const index of indicesResolveIndexResponse.indices) {
        if (!seenIndices.has(index.name)) {
          indicesWithoutDatastreamsOrAliases.add(index.name);
        }
      }
      const indexNamePartRootNode = (0, _utils.buildTree)([...indicesWithoutDatastreamsOrAliases, ...dataStreamsAndAliases]);
      const constructedIndexPatterns = (0, _utils.getIndexPatterns)(indexNamePartRootNode, {
        ignoreDigitParts: true
      });
      const indexPatterns = new Set();

      // Add any index patterns that could be constructed from the indices
      for (const indexPattern of constructedIndexPatterns.indexPatterns) {
        indexPatterns.add(indexPattern);
      }

      // Add any remaining indices that did not match any patterns
      for (const remainingIndex of constructedIndexPatterns.remainingIndices) {
        indexPatterns.add(remainingIndex);
      }
      return new _langgraph.Command({
        update: {
          indexPatterns: Array.from(indexPatterns)
        }
      });
    } catch (error) {
      return new _langgraph.Command({
        update: {
          indexPatterns: []
        }
      });
    }
  };
};
exports.fetchIndexPatterns = fetchIndexPatterns;