"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addNoteToTimelineMiddleware = void 0;
var _fp = require("lodash/fp");
var _app = require("../../../common/store/app");
var _actions = require("../../../common/store/app/actions");
var _actions2 = require("../actions");
var _api = require("../../containers/notes/api");
var _selectors = require("../selectors");
var i18n = _interopRequireWildcard(require("../../pages/translations"));
var _helpers = require("./helpers");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const timelineNoteActionsType = new Set([_actions2.addNote.type, _actions2.addNoteToEvent.type]);
function isNoteAction(action) {
  return timelineNoteActionsType.has(action.type);
}
function isAddNoteToEventAction(action) {
  return action.type === _actions2.addNoteToEvent.type;
}
const addNoteToTimelineMiddleware = kibana => store => next => async action => {
  // perform the action
  const ret = next(action);
  if (isNoteAction(action)) {
    const {
      id: localTimelineId,
      noteId: localNoteId
    } = action.payload;
    const notes = _app.appSelectors.selectNotesByIdSelector(store.getState());
    store.dispatch((0, _actions2.startTimelineSaving)({
      id: localTimelineId
    }));
    try {
      // In case a note is being added to an unsaved timeline, we need to make sure
      // the timeline has been saved or is in draft state. Otherwise, `timelineId` will be `null`
      // and we're creating orphaned notes.
      const timeline = await (0, _helpers.ensureTimelineIsSaved)({
        localTimelineId,
        timeline: (0, _selectors.selectTimelineById)(store.getState(), localTimelineId),
        store
      });
      if (!timeline.savedObjectId) {
        throw new Error('Cannot create note without a timelineId');
      }
      const result = await (0, _api.persistNote)({
        noteId: null,
        version: null,
        note: {
          eventId: 'eventId' in action.payload ? action.payload.eventId : undefined,
          note: getNoteText(localNoteId, notes),
          timelineId: timeline.savedObjectId
        }
      });
      const response = (0, _fp.get)('data.persistNote', result);
      if (response.code === 403) {
        store.dispatch((0, _actions2.showCallOutUnauthorizedMsg)());
      }
      (0, _helpers.refreshTimelines)(store.getState());
      await store.dispatch((0, _actions.updateNote)({
        note: {
          ...notes[localNoteId],
          created: response.note.updated != null ? new Date(response.note.updated) : notes[localNoteId].created,
          user: response.note.updatedBy != null ? response.note.updatedBy : notes[localNoteId].user,
          saveObjectId: response.note.noteId,
          version: response.note.version
        }
      }));
      const currentTimeline = (0, _selectors.selectTimelineById)(store.getState(), localTimelineId);

      // Automatically pin an associated event if it's not pinned yet
      if (isAddNoteToEventAction(action)) {
        const isEventPinned = currentTimeline.pinnedEventIds[action.payload.eventId] === true;
        if (!isEventPinned) {
          await store.dispatch((0, _actions2.pinEvent)({
            id: localTimelineId,
            eventId: action.payload.eventId
          }));
        }
      }
    } catch (error) {
      var _error$message;
      kibana.notifications.toasts.addDanger({
        title: i18n.UPDATE_TIMELINE_ERROR_TITLE,
        text: (_error$message = error === null || error === void 0 ? void 0 : error.message) !== null && _error$message !== void 0 ? _error$message : i18n.UPDATE_TIMELINE_ERROR_TEXT
      });
    } finally {
      store.dispatch((0, _actions2.endTimelineSaving)({
        id: localTimelineId
      }));
    }
  }
  return ret;
};
exports.addNoteToTimelineMiddleware = addNoteToTimelineMiddleware;
const getNoteText = (noteId, notes) => {
  if (noteId != null) {
    return notes[noteId].note;
  }
  return '';
};