"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.processVersionedRouter = exports.extractVersionedResponses = exports.extractVersionedResponse = exports.extractVersionedRequestBody = exports.extractVersionedRequestBodies = void 0;
var _coreHttpRouterServerInternal = require("@kbn/core-http-router-server-internal");
var _extract_authz_description = require("./extract_authz_description");
var _util = require("./util");
var _common = require("./oas_converter/common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const processVersionedRouter = (appRouter, converter, getOpId, filters) => {
  const routes = (0, _util.prepareRoutes)(appRouter.getRoutes(), filters);
  const paths = {};
  for (const route of routes) {
    const pathParams = (0, _util.getPathParameters)(route.path);
    let parameters = [];
    let handler;
    let version;
    let versions = _coreHttpRouterServerInternal.versionHandlerResolvers.sort(route.handlers.map(({
      options: {
        version: v
      }
    }) => v), route.options.access);
    if (filters !== null && filters !== void 0 && filters.version) {
      const versionIdx = versions.indexOf(filters.version);
      if (versionIdx === -1) return {
        paths
      };
      versions = versions.slice(0, versionIdx + 1);
      handler = route.handlers.find(({
        options: {
          version: v
        }
      }) => v === filters.version);
      version = filters.version;
    } else {
      version = _coreHttpRouterServerInternal.versionHandlerResolvers.newest(versions, route.options.access);
      handler = route.handlers.find(({
        options: {
          version: v
        }
      }) => v === version);
    }
    if (!handler) return {
      paths
    };
    const schemas = extractValidationSchemaFromVersionedHandler(handler);
    try {
      var _route$options$descri, _extractValidationSch, _extractValidationSch2, _route$options$option, _route$options$summar, _route$options$option2, _route$options$option3;
      if (schemas) {
        var _schemas$request, _schemas$request2;
        /**
         * Note: for a given route we accept that route params and query params remain BWC
         *       so we only take the latest version of the params and query params, we also
         *       assume at this point that we are generating for serverless.
         */
        const reqParams = (_schemas$request = schemas.request) === null || _schemas$request === void 0 ? void 0 : _schemas$request.params;
        let pathObjects = [];
        let queryObjects = [];
        if (reqParams) {
          pathObjects = converter.convertPathParameters(reqParams, pathParams);
        }
        const reqQuery = (_schemas$request2 = schemas.request) === null || _schemas$request2 === void 0 ? void 0 : _schemas$request2.query;
        if (reqQuery) {
          queryObjects = converter.convertQuery(reqQuery);
        }
        parameters = [...(0, _util.getXsrfHeaderForMethod)(route.method, route.options.options), ...pathObjects, ...queryObjects];
      }
      parameters = [...(route.options.access === 'internal' ? [(0, _util.getVersionedHeaderParam)(version, versions)] : [])].concat(...parameters);
      let description = `${(_route$options$descri = route.options.description) !== null && _route$options$descri !== void 0 ? _route$options$descri : ''}`;
      if (route.options.security) {
        const authzDescription = (0, _extract_authz_description.extractAuthzDescription)(route.options.security);
        description += `${route.options.description && authzDescription ? '<br/><br/>' : ''}${authzDescription !== null && authzDescription !== void 0 ? authzDescription : ''}`;
      }
      const hasBody = Boolean((_extractValidationSch = extractValidationSchemaFromVersionedHandler(handler)) === null || _extractValidationSch === void 0 ? void 0 : (_extractValidationSch2 = _extractValidationSch.request) === null || _extractValidationSch2 === void 0 ? void 0 : _extractValidationSch2.body);
      const contentType = (0, _util.extractContentType)((_route$options$option = route.options.options) === null || _route$options$option === void 0 ? void 0 : _route$options$option.body);
      // If any handler is deprecated we show deprecated: true in the spec
      const hasDeprecations = route.handlers.some(({
        options
      }) => {
        var _options$options;
        return !!((_options$options = options.options) !== null && _options$options !== void 0 && _options$options.deprecated);
      });
      const hasVersionFilter = Boolean(filters === null || filters === void 0 ? void 0 : filters.version);
      const operation = {
        summary: (_route$options$summar = route.options.summary) !== null && _route$options$summar !== void 0 ? _route$options$summar : '',
        tags: (_route$options$option2 = route.options.options) !== null && _route$options$option2 !== void 0 && _route$options$option2.tags ? (0, _util.extractTags)(route.options.options.tags) : [],
        ...(description ? {
          description
        } : {}),
        ...(hasDeprecations ? {
          deprecated: true
        } : {}),
        ...(route.options.discontinued ? {
          'x-discontinued': route.options.discontinued
        } : {}),
        requestBody: hasBody ? {
          content: hasVersionFilter ? extractVersionedRequestBody(handler, route.options.access, converter, contentType) : extractVersionedRequestBodies(route, converter, contentType)
        } : undefined,
        responses: hasVersionFilter ? extractVersionedResponse(handler, route.options.access, converter, contentType) : extractVersionedResponses(route, converter, contentType),
        parameters,
        operationId: getOpId({
          path: route.path,
          method: route.method
        })
      };
      (0, _util.setXState)((_route$options$option3 = route.options.options) === null || _route$options$option3 === void 0 ? void 0 : _route$options$option3.availability, operation);
      const path = {
        [route.method]: operation
      };
      (0, _util.assignToPaths)(paths, route.path, path);
    } catch (e) {
      // Enrich the error message with a bit more context
      e.message = `Error generating OpenAPI for route '${route.path}' using newest version '${version}': ${e.message}`;
      throw e;
    }
  }
  return {
    paths
  };
};
exports.processVersionedRouter = processVersionedRouter;
const extractVersionedRequestBodies = (route, converter, contentType) => {
  return route.handlers.reduce((acc, handler) => {
    return {
      ...acc,
      ...extractVersionedRequestBody(handler, route.options.access, converter, contentType)
    };
  }, {});
};
exports.extractVersionedRequestBodies = extractVersionedRequestBodies;
const extractVersionedRequestBody = (handler, access, converter, contentType) => {
  const schemas = extractValidationSchemaFromVersionedHandler(handler);
  if (!(schemas !== null && schemas !== void 0 && schemas.request)) return {};
  const schema = converter.convert(schemas.request.body);
  return {
    [(0, _util.getVersionedContentTypeString)(handler.options.version, access, contentType)]: {
      schema
    }
  };
};
exports.extractVersionedRequestBody = extractVersionedRequestBody;
const extractVersionedResponse = (handler, access, converter, contentType) => {
  const schemas = extractValidationSchemaFromVersionedHandler(handler);
  if (!(schemas !== null && schemas !== void 0 && schemas.response)) return {};
  const result = {};
  const {
    unsafe,
    ...responses
  } = schemas.response;
  for (const [statusCode, responseSchema] of Object.entries(responses)) {
    var _result$statusCode;
    let newContent;
    if (responseSchema.body) {
      const maybeSchema = (0, _coreHttpRouterServerInternal.unwrapVersionedResponseBodyValidation)(responseSchema.body);
      const schema = converter.convert(maybeSchema);
      const contentTypeString = (0, _util.getVersionedContentTypeString)(handler.options.version, access, responseSchema.bodyContentType ? [responseSchema.bodyContentType] : contentType);
      newContent = {
        [contentTypeString]: {
          schema
        }
      };
    }
    result[statusCode] = {
      ...result[statusCode],
      description: responseSchema.description,
      ...(0, _util.mergeResponseContent)(((_result$statusCode = result[statusCode]) !== null && _result$statusCode !== void 0 ? _result$statusCode : {}).content, newContent)
    };
  }
  return result;
};
exports.extractVersionedResponse = extractVersionedResponse;
const mergeDescriptions = (existing, toAppend) => {
  if (!(0, _common.isReferenceObject)(toAppend) && toAppend.description) {
    return existing !== null && existing !== void 0 && existing.length ? `${existing}\n${toAppend.description}` : toAppend.description;
  }
  return existing;
};
const mergeVersionedResponses = (a, b) => {
  const result = Object.assign({}, a);
  for (const [statusCode, responseContent] of Object.entries(b)) {
    var _ref;
    const existing = (_ref = result[statusCode]) !== null && _ref !== void 0 ? _ref : {};
    result[statusCode] = {
      ...result[statusCode],
      description: mergeDescriptions(existing.description, responseContent),
      content: Object.assign({}, existing.content, responseContent.content)
    };
  }
  return result;
};
const extractVersionedResponses = (route, converter, contentType) => {
  return route.handlers.reduce((acc, handler) => {
    const responses = extractVersionedResponse(handler, route.options.access, converter, contentType);
    return mergeVersionedResponses(acc, responses);
  }, {});
};
exports.extractVersionedResponses = extractVersionedResponses;
const extractValidationSchemaFromVersionedHandler = handler => {
  if (handler.options.validate === false) return undefined;
  if (typeof handler.options.validate === 'function') return handler.options.validate();
  return handler.options.validate;
};