"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ObservabilityAIAssistantClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _boom = require("@hapi/boom");
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _uuid = require("uuid");
var _inferenceTracing = require("@kbn/inference-tracing");
var _inferenceCommon = require("@kbn/inference-common");
var _lockManager = require("@kbn/lock-manager");
var _ = require("..");
var _conversation_complete = require("../../../common/conversation_complete");
var _convert_messages_for_inference = require("../../../common/convert_messages_for_inference");
var _types = require("../../../common/types");
var _context = require("../../functions/context/context");
var _get_access_query = require("../util/get_access_query");
var _get_system_message_from_instructions = require("../util/get_system_message_from_instructions");
var _fail_on_non_existing_function_call = require("./operators/fail_on_non_existing_function_call");
var _get_context_function_request_if_needed = require("./get_context_function_request_if_needed");
var _continue_conversation = require("./operators/continue_conversation");
var _convert_inference_events_to_streaming_events = require("./operators/convert_inference_events_to_streaming_events");
var _extract_messages = require("./operators/extract_messages");
var _get_generated_title = require("./operators/get_generated_title");
var _run_startup_migrations = require("../startup_migrations/run_startup_migrations");
var _inference_endpoint = require("../inference_endpoint");
var _reindex_knowledge_base = require("../knowledge_base_service/reindex_knowledge_base");
var _populate_missing_semantic_text_fields = require("../startup_migrations/populate_missing_semantic_text_fields");
var _create_or_update_knowledge_base_index_assets = require("../index_assets/create_or_update_knowledge_base_index_assets");
var _get_inference_id_from_write_index = require("../knowledge_base_service/get_inference_id_from_write_index");
var _preconfigured_inference_ids = require("../../../common/preconfigured_inference_ids");
var _add_anonymization_data = require("./operators/add_anonymization_data");
var _conversation_delete = require("../../analytics/conversation_delete");
var _conversation_duplicate = require("../../analytics/conversation_duplicate");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_FUNCTION_CALLS = 8;
class ObservabilityAIAssistantClient {
  constructor(dependencies) {
    (0, _defineProperty2.default)(this, "getConversationWithMetaFields", async conversationId => {
      const response = await this.dependencies.esClient.asInternalUser.search({
        index: _.resourceNames.writeIndexAlias.conversations,
        query: {
          bool: {
            filter: [...(0, _get_access_query.getAccessQuery)({
              user: this.dependencies.user,
              namespace: this.dependencies.namespace
            }), {
              term: {
                'conversation.id': conversationId
              }
            }]
          }
        },
        size: 1,
        terminate_after: 1
      });
      return response.hits.hits[0];
    });
    (0, _defineProperty2.default)(this, "getConversationUpdateValues", lastUpdated => {
      return {
        conversation: {
          last_updated: lastUpdated
        },
        user: this.dependencies.user,
        namespace: this.dependencies.namespace
      };
    });
    (0, _defineProperty2.default)(this, "isConversationOwnedByUser", conversation => {
      const user = this.dependencies.user;
      if (!conversation.user || !user) {
        return false;
      }
      return conversation.user.id && user.id ? conversation.user.id === user.id : conversation.user.name === user.name;
    });
    (0, _defineProperty2.default)(this, "get", async conversationId => {
      const conversation = await this.getConversationWithMetaFields(conversationId);
      if (!conversation) {
        throw (0, _boom.notFound)();
      }
      return conversation._source;
    });
    (0, _defineProperty2.default)(this, "delete", async conversationId => {
      const conversation = await this.getConversationWithMetaFields(conversationId);
      if (!conversation) {
        throw (0, _boom.notFound)();
      }
      if (!this.isConversationOwnedByUser(conversation._source)) {
        throw (0, _boom.forbidden)('Deleting a conversation is only allowed for the owner of the conversation.');
      }
      await this.dependencies.esClient.asInternalUser.delete({
        id: conversation._id,
        index: conversation._index,
        refresh: true
      });
      this.dependencies.analytics.reportEvent(_conversation_delete.conversationDeleteEvent.eventType, {});
    });
    (0, _defineProperty2.default)(this, "complete", ({
      functionClient,
      connectorId,
      simulateFunctionCalling = false,
      userInstructions: apiUserInstructions = [],
      messages: initialMessages,
      signal,
      persist,
      kibanaPublicUrl,
      isPublic,
      title: predefinedTitle,
      conversationId: predefinedConversationId,
      disableFunctions = false
    }) => {
      return (0, _inferenceTracing.withInferenceSpan)('run_tools', () => {
        const isConversationUpdate = persist && !!predefinedConversationId;
        const conversationId = persist ? predefinedConversationId || (0, _uuid.v4)() : '';
        if (persist && !isConversationUpdate && kibanaPublicUrl) {
          functionClient.registerInstruction(`This conversation will be persisted in Kibana and available at this url: ${kibanaPublicUrl}/app/observabilityAIAssistant/conversations/${conversationId}.`);
        }
        const kbUserInstructions$ = (0, _rxjs.from)(this.getKnowledgeBaseUserInstructions()).pipe((0, _rxjs.shareReplay)());

        // if it is:
        // - a new conversation
        // - no predefined title is given
        // - we need to store the conversation
        // we generate a title
        // if not, we complete with an empty string
        const title$ = predefinedTitle || isConversationUpdate || !persist ? (0, _rxjs.of)(predefinedTitle || '').pipe((0, _rxjs.shareReplay)()) : (0, _get_generated_title.getGeneratedTitle)({
          messages: initialMessages,
          logger: this.dependencies.logger,
          scopes: this.dependencies.scopes,
          chat: (name, chatParams) => (0, _inferenceTracing.withInferenceSpan)('get_title', () => this.chat(name, {
            ...chatParams,
            simulateFunctionCalling,
            connectorId,
            signal,
            stream: false
          }))
        }).pipe((0, _rxjs.shareReplay)());
        const systemMessage$ = kbUserInstructions$.pipe((0, _rxjs.map)(kbUserInstructions => (0, _get_system_message_from_instructions.getSystemMessageFromInstructions)({
          applicationInstructions: functionClient.getInstructions(),
          kbUserInstructions,
          apiUserInstructions,
          availableFunctionNames: disableFunctions ? [] : functionClient.getFunctions().map(fn => fn.definition.name)
        })), (0, _rxjs.shareReplay)());
        const connector$ = (0, _rxjs.defer)(() => (0, _rxjs.from)(this.dependencies.actionsClient.get({
          id: connectorId,
          throwIfSystemAction: true
        })).pipe((0, _rxjs.catchError)(error => {
          this.dependencies.logger.debug(`Failed to fetch connector for analytics: ${error.message}`);
          return (0, _rxjs.of)(undefined);
        }), (0, _rxjs.shareReplay)()));

        // we continue the conversation here, after resolving both the materialized
        // messages and the knowledge base instructions
        const nextEvents$ = (0, _rxjs.forkJoin)([systemMessage$, kbUserInstructions$, connector$]).pipe((0, _rxjs.switchMap)(([systemMessage, kbUserInstructions, connector]) => {
          // if needed, inject a context function request here
          const contextRequest = functionClient.hasFunction(_context.CONTEXT_FUNCTION_NAME) ? (0, _get_context_function_request_if_needed.getContextFunctionRequestIfNeeded)(initialMessages) : undefined;
          return (0, _inferenceTracing.withInferenceSpan)('run_tools', () => (0, _rxjs.merge)(
          // if we have added a context function request, also emit
          // the messageAdd event for it, so we can notify the consumer
          // and add it to the conversation
          ...(contextRequest ? [(0, _rxjs.of)(contextRequest)] : []), (0, _continue_conversation.continueConversation)({
            messages: [...initialMessages, ...(contextRequest ? [contextRequest.message] : [])],
            chat: (name, chatParams) => {
              // inject a chat function with predefined parameters
              return this.chat(name, {
                systemMessage,
                ...chatParams,
                signal,
                simulateFunctionCalling,
                connectorId,
                stream: true
              });
            },
            // start out with the max number of function calls
            functionCallsLeft: MAX_FUNCTION_CALLS,
            functionClient,
            kbUserInstructions,
            apiUserInstructions,
            signal,
            logger: this.dependencies.logger,
            disableFunctions,
            connectorId,
            simulateFunctionCalling,
            analytics: this.dependencies.analytics,
            connector,
            scopes: this.dependencies.scopes
          })));
        }), (0, _rxjs.shareReplay)());
        const conversationWithMetaFields$ = (0, _rxjs.from)(this.getConversationWithMetaFields(conversationId)).pipe((0, _rxjs.switchMap)(conversation => {
          if (isConversationUpdate && !conversation) {
            return (0, _rxjs.throwError)(() => (0, _conversation_complete.createConversationNotFoundError)());
          }
          if (conversation !== null && conversation !== void 0 && conversation._source && !this.isConversationOwnedByUser(conversation._source)) {
            return (0, _rxjs.throwError)(() => new Error('Cannot update conversation that is not owned by the user'));
          }
          return (0, _rxjs.of)(conversation);
        }));
        const output$ = conversationWithMetaFields$.pipe((0, _rxjs.switchMap)(conversation => {
          return (0, _rxjs.merge)(
          // get all the events from continuing the conversation
          nextEvents$,
          // wait until all dependencies have completed
          (0, _rxjs.forkJoin)([
          // get just the new messages
          nextEvents$.pipe((0, _extract_messages.extractMessages)()),
          // get just the title, and drop the token count events
          title$.pipe((0, _rxjs.filter)(value => typeof value === 'string')), systemMessage$]).pipe((0, _rxjs.switchMap)(([addedMessages, title, systemMessage]) => {
            return nextEvents$.pipe((0, _add_anonymization_data.addAnonymizationData)(initialMessages.concat(addedMessages)), (0, _rxjs.switchMap)(deanonymizedMessages => {
              var _lastMessage$message$;
              const lastMessage = (0, _lodash.last)(deanonymizedMessages);

              // if a function request is at the very end, close the stream to consumer
              // without persisting or updating the conversation. we need to wait
              // on the function response to have a valid conversation
              const isFunctionRequest = !!(lastMessage !== null && lastMessage !== void 0 && (_lastMessage$message$ = lastMessage.message.function_call) !== null && _lastMessage$message$ !== void 0 && _lastMessage$message$.name);
              if (!persist || isFunctionRequest) {
                return (0, _rxjs.of)();
              }
              if (isConversationUpdate && conversation) {
                var _conversation$_source;
                return (0, _rxjs.from)(this.update(conversationId, (0, _lodash.merge)({},
                // base conversation without messages
                (0, _lodash.omit)(conversation._source, 'messages'),
                // update messages and system message
                {
                  messages: deanonymizedMessages,
                  systemMessage
                },
                // update title
                {
                  conversation: {
                    title: title || ((_conversation$_source = conversation._source) === null || _conversation$_source === void 0 ? void 0 : _conversation$_source.conversation.title)
                  }
                }))).pipe((0, _rxjs.map)(conversationUpdated => {
                  return {
                    conversation: conversationUpdated.conversation,
                    type: _conversation_complete.StreamingChatResponseEventType.ConversationUpdate
                  };
                }));
              }
              return (0, _rxjs.from)(this.create({
                '@timestamp': new Date().toISOString(),
                conversation: {
                  title,
                  id: conversationId
                },
                public: !!isPublic,
                labels: {},
                numeric_labels: {},
                systemMessage,
                messages: deanonymizedMessages,
                archived: false
              })).pipe((0, _rxjs.map)(conversationCreated => {
                return {
                  conversation: conversationCreated.conversation,
                  type: _conversation_complete.StreamingChatResponseEventType.ConversationCreate
                };
              }));
            }));
          })));
        }));
        return output$.pipe((0, _rxjs.catchError)(error => {
          this.dependencies.logger.error(error);
          return (0, _rxjs.throwError)(() => error);
        }), (0, _rxjs.tap)(event => {
          switch (event.type) {
            case _conversation_complete.StreamingChatResponseEventType.MessageAdd:
              this.dependencies.logger.debug(() => `Added message: ${JSON.stringify(event.message)}`);
              break;
            case _conversation_complete.StreamingChatResponseEventType.ConversationCreate:
              this.dependencies.logger.debug(() => `Created conversation: ${JSON.stringify(event.conversation)}`);
              break;
            case _conversation_complete.StreamingChatResponseEventType.ConversationUpdate:
              this.dependencies.logger.debug(() => `Updated conversation: ${JSON.stringify(event.conversation)}`);
              break;
          }
        }), (0, _rxjs.shareReplay)());
      });
    });
    (0, _defineProperty2.default)(this, "find", async options => {
      const response = await this.dependencies.esClient.asInternalUser.search({
        index: _.resourceNames.writeIndexAlias.conversations,
        allow_no_indices: true,
        query: {
          bool: {
            filter: [...(0, _get_access_query.getAccessQuery)({
              user: this.dependencies.user,
              namespace: this.dependencies.namespace
            })]
          }
        },
        sort: {
          '@timestamp': 'desc'
        },
        size: 100
      });
      return response.hits.hits.map(hit => hit._source);
    });
    (0, _defineProperty2.default)(this, "update", async (conversationId, conversation) => {
      const persistedConversation = await this.getConversationWithMetaFields(conversationId);
      if (!persistedConversation) {
        throw (0, _boom.notFound)();
      }
      if (!this.isConversationOwnedByUser(persistedConversation._source)) {
        throw (0, _boom.forbidden)('Updating a conversation is only allowed for the owner of the conversation.');
      }
      const updatedConversation = (0, _lodash.merge)({}, conversation, this.getConversationUpdateValues(new Date().toISOString()));
      await this.dependencies.esClient.asInternalUser.update({
        id: persistedConversation._id,
        index: persistedConversation._index,
        doc: updatedConversation,
        refresh: true
      });
      return updatedConversation;
    });
    (0, _defineProperty2.default)(this, "create", async conversation => {
      const now = new Date().toISOString();
      const createdConversation = (0, _lodash.merge)({}, conversation, {
        '@timestamp': now,
        conversation: {
          id: conversation.conversation.id || (0, _uuid.v4)()
        }
      }, this.getConversationUpdateValues(now));
      await this.dependencies.esClient.asInternalUser.index({
        index: _.resourceNames.writeIndexAlias.conversations,
        document: createdConversation,
        refresh: true
      });
      return createdConversation;
    });
    (0, _defineProperty2.default)(this, "updatePartial", async ({
      conversationId,
      updates
    }) => {
      const conversation = await this.get(conversationId);
      if (!conversation) {
        throw (0, _boom.notFound)();
      }
      const updatedConversation = (0, _lodash.merge)({}, conversation, {
        ...(updates.public !== undefined && {
          public: updates.public
        }),
        ...(updates.archived !== undefined && {
          archived: updates.archived
        })
      });
      return this.update(conversationId, updatedConversation);
    });
    (0, _defineProperty2.default)(this, "duplicateConversation", async conversationId => {
      const conversation = await this.getConversationWithMetaFields(conversationId);
      if (!conversation) {
        throw (0, _boom.notFound)();
      }
      const _source = conversation._source;
      const duplicatedConversation = await this.create({
        ..._source,
        conversation: {
          ..._source.conversation,
          id: (0, _uuid.v4)()
        },
        public: false,
        archived: false
      });
      this.dependencies.analytics.reportEvent(_conversation_duplicate.conversationDuplicateEvent.eventType, {});
      return duplicatedConversation;
    });
    (0, _defineProperty2.default)(this, "recall", async ({
      queries,
      categories,
      limit
    }) => {
      var _this$dependencies$kn;
      return ((_this$dependencies$kn = this.dependencies.knowledgeBaseService) === null || _this$dependencies$kn === void 0 ? void 0 : _this$dependencies$kn.recall({
        namespace: this.dependencies.namespace,
        user: this.dependencies.user,
        queries,
        categories,
        esClient: this.dependencies.esClient,
        uiSettingsClient: this.dependencies.uiSettingsClient,
        limit
      })) || [];
    });
    (0, _defineProperty2.default)(this, "getInferenceEndpointsForEmbedding", () => {
      return this.dependencies.knowledgeBaseService.getInferenceEndpointsForEmbedding();
    });
    (0, _defineProperty2.default)(this, "getKnowledgeBaseStatus", () => {
      return this.dependencies.knowledgeBaseService.getModelStatus();
    });
    (0, _defineProperty2.default)(this, "setupKnowledgeBase", async (nextInferenceId, waitUntilComplete = false) => {
      const {
        esClient,
        core,
        logger
      } = this.dependencies;
      logger.debug(`Setting up knowledge base with inference_id: ${nextInferenceId}`);
      const currentInferenceId = await (0, _get_inference_id_from_write_index.getInferenceIdFromWriteIndex)(esClient).catch(() => {
        logger.debug(`Current KB write index does not have an inference_id. This is to be expected for indices created before 8.16`);
        return undefined;
      });
      if (currentInferenceId === nextInferenceId) {
        logger.debug('Inference ID is unchanged. No need to re-index knowledge base.');
        (0, _inference_endpoint.warmupModel)({
          esClient,
          logger,
          inferenceId: nextInferenceId
        }).catch(() => {});
        return {
          reindex: false,
          currentInferenceId,
          nextInferenceId
        };
      }
      await (0, _create_or_update_knowledge_base_index_assets.createOrUpdateKnowledgeBaseIndexAssets)({
        core: this.dependencies.core,
        logger: this.dependencies.logger,
        inferenceId: nextInferenceId
      });
      const kbSetupPromise = (0, _inference_endpoint.waitForKbModel)({
        core: this.dependencies.core,
        esClient,
        logger,
        config: this.dependencies.config,
        inferenceId: nextInferenceId
      }).then(async () => {
        logger.info(`Inference ID has changed from "${currentInferenceId}" to "${nextInferenceId}". Re-indexing knowledge base.`);
        await (0, _reindex_knowledge_base.reIndexKnowledgeBaseWithLock)({
          core,
          logger,
          esClient
        });
        await (0, _populate_missing_semantic_text_fields.populateMissingSemanticTextFieldWithLock)({
          core,
          logger,
          config: this.dependencies.config,
          esClient: this.dependencies.esClient
        });

        // If the inference ID switched to a preconfigured inference endpoint, delete the legacy custom inference endpoint if it exists.
        if (currentInferenceId === _preconfigured_inference_ids.LEGACY_CUSTOM_INFERENCE_ID) {
          void (0, _inference_endpoint.deleteInferenceEndpoint)({
            esClient,
            logger,
            inferenceId: _preconfigured_inference_ids.LEGACY_CUSTOM_INFERENCE_ID
          });
        }
      }).catch(e => {
        if ((0, _lockManager.isLockAcquisitionError)(e)) {
          logger.info(e.message);
        } else {
          logger.error(`Failed to setup knowledge base with inference_id: ${nextInferenceId}. Error: ${e.message}`);
          logger.debug(e);
        }
      });
      if (waitUntilComplete) {
        await kbSetupPromise;
      }
      return {
        reindex: true,
        currentInferenceId,
        nextInferenceId
      };
    });
    (0, _defineProperty2.default)(this, "warmupKbModel", inferenceId => {
      const {
        esClient,
        logger
      } = this.dependencies;
      logger.debug(`Warming up model for for inference ID: ${inferenceId}`);
      (0, _inference_endpoint.warmupModel)({
        esClient,
        logger,
        inferenceId
      }).catch(() => {});
      return;
    });
    (0, _defineProperty2.default)(this, "reIndexKnowledgeBaseWithLock", () => {
      return (0, _reindex_knowledge_base.reIndexKnowledgeBaseWithLock)({
        core: this.dependencies.core,
        esClient: this.dependencies.esClient,
        logger: this.dependencies.logger
      });
    });
    (0, _defineProperty2.default)(this, "runStartupMigrations", () => {
      return (0, _run_startup_migrations.runStartupMigrations)({
        core: this.dependencies.core,
        logger: this.dependencies.logger,
        config: this.dependencies.config
      });
    });
    (0, _defineProperty2.default)(this, "addUserInstruction", async ({
      entry
    }) => {
      // for now we want to limit the number of user instructions to 1 per user
      // if a user instruction already exists for the user, we get the id and update it

      const existingId = await this.dependencies.knowledgeBaseService.getPersonalUserInstructionId({
        isPublic: entry.public,
        namespace: this.dependencies.namespace,
        user: this.dependencies.user
      });
      if (existingId) {
        var _this$dependencies$us;
        this.dependencies.logger.debug(`Updating user instruction. id = "${existingId}", user = "${(_this$dependencies$us = this.dependencies.user) === null || _this$dependencies$us === void 0 ? void 0 : _this$dependencies$us.name}"`);
        entry.id = existingId;
      } else {
        var _this$dependencies$us2;
        this.dependencies.logger.debug(`Creating user instruction. id = "${entry.id}", user = "${(_this$dependencies$us2 = this.dependencies.user) === null || _this$dependencies$us2 === void 0 ? void 0 : _this$dependencies$us2.name}"`);
      }
      return this.dependencies.knowledgeBaseService.addEntry({
        namespace: this.dependencies.namespace,
        user: this.dependencies.user,
        entry: {
          ...entry,
          type: _types.KnowledgeBaseType.UserInstruction,
          labels: {},
          role: _types.KnowledgeBaseEntryRole.UserEntry
        }
      });
    });
    (0, _defineProperty2.default)(this, "addKnowledgeBaseEntry", async ({
      entry
    }) => {
      return this.dependencies.knowledgeBaseService.addEntry({
        namespace: this.dependencies.namespace,
        user: this.dependencies.user,
        entry: {
          ...entry,
          type: _types.KnowledgeBaseType.Contextual
        }
      });
    });
    (0, _defineProperty2.default)(this, "addKnowledgeBaseBulkEntries", async ({
      entries
    }) => {
      return this.dependencies.knowledgeBaseService.addBulkEntries({
        entries: entries.map(entry => ({
          ...entry,
          type: _types.KnowledgeBaseType.Contextual
        })),
        user: this.dependencies.user,
        namespace: this.dependencies.namespace
      });
    });
    (0, _defineProperty2.default)(this, "getKnowledgeBaseEntries", async ({
      query,
      sortBy,
      sortDirection
    }) => {
      return this.dependencies.knowledgeBaseService.getEntries({
        query,
        sortBy,
        sortDirection,
        namespace: this.dependencies.namespace
      });
    });
    (0, _defineProperty2.default)(this, "deleteKnowledgeBaseEntry", async id => {
      return this.dependencies.knowledgeBaseService.deleteEntry({
        id
      });
    });
    (0, _defineProperty2.default)(this, "getKnowledgeBaseUserInstructions", async () => {
      return this.dependencies.knowledgeBaseService.getUserInstructions(this.dependencies.namespace, this.dependencies.user);
    });
    this.dependencies = dependencies;
  }
  chat(name, {
    systemMessage,
    messages,
    connectorId,
    functions,
    functionCall,
    signal,
    simulateFunctionCalling,
    stream
  }) {
    let tools;
    let toolChoice;
    if (functions !== null && functions !== void 0 && functions.length) {
      tools = functions.reduce((acc, fn) => {
        acc[fn.name] = {
          description: fn.description,
          schema: fn.parameters
        };
        return acc;
      }, {});
      toolChoice = functionCall ? {
        function: functionCall
      } : _inferenceCommon.ToolChoiceType.auto;
    }
    const options = {
      connectorId,
      system: systemMessage,
      messages: (0, _convert_messages_for_inference.convertMessagesForInference)(messages, this.dependencies.logger),
      toolChoice,
      tools,
      functionCalling: simulateFunctionCalling ? 'simulated' : 'auto',
      metadata: {
        connectorTelemetry: {
          pluginId: 'observability_ai_assistant'
        }
      }
    };
    this.dependencies.logger.debug(() => `Options for inference client for name: "${name}" before anonymization: ${JSON.stringify(options)}`);
    if (stream) {
      return (0, _rxjs.defer)(() => this.dependencies.inferenceClient.chatComplete({
        ...options,
        temperature: 0.25,
        maxRetries: 0,
        stream: true
      })).pipe((0, _convert_inference_events_to_streaming_events.convertInferenceEventsToStreamingEvents)(), (0, _fail_on_non_existing_function_call.failOnNonExistingFunctionCall)({
        functions
      }), (0, _rxjs.tap)(event => {
        if (event.type === _conversation_complete.StreamingChatResponseEventType.ChatCompletionChunk) {
          this.dependencies.logger.trace(() => `Received chunk: ${JSON.stringify(event.message)}`);
        }
      }), (0, _rxjs.shareReplay)());
    } else {
      return this.dependencies.inferenceClient.chatComplete({
        ...options,
        messages: (0, _convert_messages_for_inference.convertMessagesForInference)(messages, this.dependencies.logger),
        temperature: 0.25,
        maxRetries: 0,
        stream: false
      });
    }
  }
}
exports.ObservabilityAIAssistantClient = ObservabilityAIAssistantClient;