"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.outputIdToUuid = outputIdToUuid;
exports.outputSavedObjectToOutput = outputSavedObjectToOutput;
exports.outputService = void 0;
var _uuid = require("uuid");
var _lodash = _interopRequireWildcard(require("lodash"));
var _jsYaml = require("js-yaml");
var _fastDeepEqual = _interopRequireDefault(require("fast-deep-equal"));
var _fp = require("lodash/fp");
var _server = require("@kbn/core/server");
var _pMap = _interopRequireDefault(require("p-map"));
var _output_helpers = require("../../common/services/output_helpers");
var _constants = require("../constants");
var _constants2 = require("../../common/constants");
var _services = require("../../common/services");
var _errors = require("../errors");
var _agent_policy = require("./agent_policy");
var _package_policy = require("./package_policy");
var _app_context = require("./app_context");
var _saved_object = require("./saved_object");
var _audit_logging = require("./audit_logging");
var _secrets = require("./secrets");
var _helpers = require("./outputs/helpers");
var _so_helpers = require("./outputs/so_helpers");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SAVED_OBJECT_TYPE = _constants.OUTPUT_SAVED_OBJECT_TYPE;
const DEFAULT_ES_HOSTS = ['http://localhost:9200'];
const fakeRequest = {
  headers: {},
  getBasePath: () => '',
  path: '/',
  route: {
    settings: {}
  },
  url: {
    href: '/'
  },
  raw: {
    req: {
      url: '/'
    }
  }
};

// differentiate
function isUUID(val) {
  return typeof val === 'string' && val.match(/[0-9a-fA-F]{8}\-[0-9a-fA-F]{4}\-[0-9a-fA-F]{4}\-[0-9a-fA-F]{4}\-[0-9a-fA-F]{12}/);
}
function outputIdToUuid(id) {
  if (isUUID(id)) {
    return id;
  }

  // UUID v5 need a namespace (uuid.DNS), changing this params will result in loosing the ability to generate predicable uuid
  return (0, _uuid.v5)(id, _uuid.v5.DNS);
}
function outputSavedObjectToOutput(so) {
  const logger = _app_context.appContextService.getLogger();
  const {
    output_id: outputId,
    ssl,
    proxy_id: proxyId,
    ...atributes
  } = so.attributes;
  let parsedSsl;
  try {
    parsedSsl = typeof ssl === 'string' ? JSON.parse(ssl) : undefined;
  } catch (e) {
    logger.warn(`Unable to parse ssl for output ${so.id}: ${e.message}`);
    logger.warn(`ssl value: ${ssl}`);
  }
  return {
    id: outputId !== null && outputId !== void 0 ? outputId : so.id,
    ...atributes,
    ...(parsedSsl ? {
      ssl: parsedSsl
    } : {}),
    ...(proxyId ? {
      proxy_id: proxyId
    } : {})
  };
}
async function getAgentPoliciesPerOutput(outputId, isDefault) {
  const internalSoClientWithoutSpaceExtension = _app_context.appContextService.getInternalUserSOClientWithoutSpaceExtension();
  let agentPoliciesKuery;
  const packagePoliciesKuery = `${_constants.PACKAGE_POLICY_SAVED_OBJECT_TYPE}.output_id:"${outputId}"`;
  if (outputId) {
    if (isDefault) {
      agentPoliciesKuery = `${_constants.LEGACY_AGENT_POLICY_SAVED_OBJECT_TYPE}.data_output_id:"${outputId}" or not ${_constants.LEGACY_AGENT_POLICY_SAVED_OBJECT_TYPE}.data_output_id:*`;
    } else {
      agentPoliciesKuery = `${_constants.LEGACY_AGENT_POLICY_SAVED_OBJECT_TYPE}.data_output_id:"${outputId}"`;
    }
  } else {
    if (isDefault) {
      agentPoliciesKuery = `not ${_constants.LEGACY_AGENT_POLICY_SAVED_OBJECT_TYPE}.data_output_id:*`;
    } else {
      return;
    }
  }

  // Get agent policies directly using output
  const directAgentPolicies = await _agent_policy.agentPolicyService.list(internalSoClientWithoutSpaceExtension, {
    kuery: agentPoliciesKuery,
    perPage: _constants2.SO_SEARCH_LIMIT,
    spaceId: '*'
  });
  const directAgentPolicyIds = directAgentPolicies === null || directAgentPolicies === void 0 ? void 0 : directAgentPolicies.items.map(policy => policy.id);

  // Get package policies using output and derive agent policies from that which
  // are not already identfied above. The IDs cannot be used as part of the kuery
  // above since the underlying saved object client .find() only filters on attributes
  const packagePolicySOs = await _package_policy.packagePolicyService.list(internalSoClientWithoutSpaceExtension, {
    kuery: packagePoliciesKuery,
    perPage: _constants2.SO_SEARCH_LIMIT,
    spaceId: '*'
  });
  const agentPolicyIdsFromPackagePolicies = [...new Set(packagePolicySOs === null || packagePolicySOs === void 0 ? void 0 : packagePolicySOs.items.reduce((acc, packagePolicy) => {
    return [...acc, ...packagePolicy.policy_ids.filter(id => !(directAgentPolicyIds !== null && directAgentPolicyIds !== void 0 && directAgentPolicyIds.includes(id)))];
  }, []))];
  const agentPoliciesFromPackagePolicies = await _agent_policy.agentPolicyService.getByIDs(internalSoClientWithoutSpaceExtension, agentPolicyIdsFromPackagePolicies);
  const agentPoliciesIndexedById = (0, _fp.indexBy)(policy => policy.id, [...directAgentPolicies.items, ...agentPoliciesFromPackagePolicies]);

  // Bulk fetch package policies with only needed fields
  if (Object.keys(agentPoliciesIndexedById).length) {
    const {
      items: packagePolicies
    } = await _package_policy.packagePolicyService.list(internalSoClientWithoutSpaceExtension, {
      fields: ['policy_ids', 'package.name'],
      kuery: [_constants2.FLEET_APM_PACKAGE, _constants2.FLEET_SYNTHETICS_PACKAGE, _constants2.FLEET_SERVER_PACKAGE].map(packageName => `${_constants.PACKAGE_POLICY_SAVED_OBJECT_TYPE}.package.name:${packageName}`).join(' or ')
    });
    for (const packagePolicy of packagePolicies) {
      for (const policyId of packagePolicy.policy_ids) {
        if (agentPoliciesIndexedById[policyId]) {
          var _agentPoliciesIndexed;
          if (!agentPoliciesIndexedById[policyId].package_policies) {
            agentPoliciesIndexedById[policyId].package_policies = [];
          }
          (_agentPoliciesIndexed = agentPoliciesIndexedById[policyId].package_policies) === null || _agentPoliciesIndexed === void 0 ? void 0 : _agentPoliciesIndexed.push(packagePolicy);
        }
      }
    }
  }
  return Object.values(agentPoliciesIndexedById);
}
async function validateLogstashOutputNotUsedInAPMPolicy(outputId, isDefault) {
  const agentPolicies = await getAgentPoliciesPerOutput(outputId, isDefault);

  // Validate no policy with APM use that policy
  if (agentPolicies) {
    for (const agentPolicy of agentPolicies) {
      if (_agent_policy.agentPolicyService.hasAPMIntegration(agentPolicy)) {
        throw new _errors.OutputInvalidError('Logstash output cannot be used with APM integration.');
      }
    }
  }
}
async function findPoliciesWithFleetServerOrSynthetics(outputId, isDefault) {
  var _agentPolicies, _agentPolicies2;
  const internalSoClientWithoutSpaceExtension = _app_context.appContextService.getInternalUserSOClientWithoutSpaceExtension();
  let agentPolicies;
  if (outputId) {
    agentPolicies = await getAgentPoliciesPerOutput(outputId, isDefault);
  } else {
    const {
      items: packagePolicies
    } = await _package_policy.packagePolicyService.list(internalSoClientWithoutSpaceExtension, {
      fields: ['policy_ids', 'package.name'],
      spaceId: '*',
      kuery: [_constants2.FLEET_APM_PACKAGE, _constants2.FLEET_SYNTHETICS_PACKAGE, _constants2.FLEET_SERVER_PACKAGE].map(packageName => `${_constants.PACKAGE_POLICY_SAVED_OBJECT_TYPE}.package.name:${packageName}`).join(' or ')
    });
    const agentPolicyIds = _lodash.default.uniq(packagePolicies.flatMap(p => p.policy_ids));
    if (agentPolicyIds.length) {
      agentPolicies = await _agent_policy.agentPolicyService.getByIDs(internalSoClientWithoutSpaceExtension, agentPolicyIds);
      for (const packagePolicy of packagePolicies) {
        for (const policyId of packagePolicy.policy_ids) {
          const agentPolicy = agentPolicies.find(p => p.id === policyId);
          if (agentPolicy) {
            var _agentPolicy$package_;
            if (!agentPolicy.package_policies) {
              agentPolicy.package_policies = [];
            }
            (_agentPolicy$package_ = agentPolicy.package_policies) === null || _agentPolicy$package_ === void 0 ? void 0 : _agentPolicy$package_.push(packagePolicy);
          }
        }
      }
    }
  }
  const policiesWithFleetServer = ((_agentPolicies = agentPolicies) === null || _agentPolicies === void 0 ? void 0 : _agentPolicies.filter(policy => _agent_policy.agentPolicyService.hasFleetServerIntegration(policy))) || [];
  const policiesWithSynthetics = ((_agentPolicies2 = agentPolicies) === null || _agentPolicies2 === void 0 ? void 0 : _agentPolicies2.filter(policy => _agent_policy.agentPolicyService.hasSyntheticsIntegration(policy))) || [];
  return {
    policiesWithFleetServer,
    policiesWithSynthetics
  };
}
function validateOutputNotUsedInPolicy(agentPolicies, dataOutputType, integrationName) {
  // Validate no policy with this integration uses that output
  for (const agentPolicy of agentPolicies) {
    throw new _errors.OutputInvalidError(`${_lodash.default.capitalize(dataOutputType)} output cannot be used with ${integrationName} integration in ${agentPolicy.name}. Please create a new Elasticsearch output.`);
  }
}
async function validateTypeChanges(esClient, id, data, originalOutput, defaultDataOutputId, fromPreconfiguration) {
  var _data$is_default;
  const internalSoClientWithoutSpaceExtension = _app_context.appContextService.getInternalUserSOClientWithoutSpaceExtension();
  const mergedIsDefault = (_data$is_default = data.is_default) !== null && _data$is_default !== void 0 ? _data$is_default : originalOutput.is_default;
  const {
    policiesWithFleetServer,
    policiesWithSynthetics
  } = await findPoliciesWithFleetServerOrSynthetics(id, mergedIsDefault);
  const agentlessPolicies = await (0, _helpers.findAgentlessPolicies)(id);
  if (data.type === _constants2.outputType.Logstash || originalOutput.type === _constants2.outputType.Logstash) {
    await validateLogstashOutputNotUsedInAPMPolicy(id, mergedIsDefault);
  }
  // prevent changing an ES output to a non-local ES output if it's used by an invalid policy
  if (originalOutput.type === _constants2.outputType.Elasticsearch && (data === null || data === void 0 ? void 0 : data.type) !== _constants2.outputType.Elasticsearch && data.type) {
    // Validate no policy with fleet server, synthetics, or agentless policies use that output
    validateOutputNotUsedInPolicy(policiesWithFleetServer, data.type, 'Fleet Server');
    validateOutputNotUsedInPolicy(policiesWithSynthetics, data.type, 'Synthetics');
    validateOutputNotUsedInPolicy(agentlessPolicies, data.type, 'agentless');
  }
  await updateAgentPoliciesDataOutputId(internalSoClientWithoutSpaceExtension, esClient, data, mergedIsDefault, defaultDataOutputId, _lodash.default.uniq([...policiesWithFleetServer, ...policiesWithSynthetics, ...agentlessPolicies]), fromPreconfiguration);
}
async function updateAgentPoliciesDataOutputId(soClient, esClient, data, isDefault, defaultDataOutputId, agentPolicies, fromPreconfiguration) {
  // if a non-local ES output is about to be updated to become default
  // and fleet server, synthetics, or agentless policies don't have
  // data_output_id set, update them to use the current default output ID
  if ((data === null || data === void 0 ? void 0 : data.type) !== _constants2.outputType.Elasticsearch && isDefault) {
    for (const policy of agentPolicies) {
      if (!policy.data_output_id) {
        await _agent_policy.agentPolicyService.update(soClient, esClient, policy.id, {
          data_output_id: defaultDataOutputId
        }, {
          force: fromPreconfiguration
        });
      }
    }
  }
}
class OutputService {
  get encryptedSoClient() {
    return _app_context.appContextService.getInternalUserSOClient(fakeRequest);
  }
  async _getDefaultDataOutputsSO() {
    const outputs = await this.encryptedSoClient.find({
      type: _constants.OUTPUT_SAVED_OBJECT_TYPE,
      searchFields: ['is_default'],
      search: 'true'
    });
    for (const output of outputs.saved_objects) {
      _audit_logging.auditLoggingService.writeCustomSoAuditLog({
        action: 'get',
        id: output.id,
        savedObjectType: _constants.OUTPUT_SAVED_OBJECT_TYPE
      });
    }
    return outputs;
  }
  async _getDefaultMonitoringOutputsSO(soClient) {
    const outputs = await this.encryptedSoClient.find({
      type: _constants.OUTPUT_SAVED_OBJECT_TYPE,
      searchFields: ['is_default_monitoring'],
      search: 'true'
    });
    for (const output of outputs.saved_objects) {
      _audit_logging.auditLoggingService.writeCustomSoAuditLog({
        action: 'get',
        id: output.id,
        savedObjectType: _constants.OUTPUT_SAVED_OBJECT_TYPE
      });
    }
    return outputs;
  }
  async _updateDefaultOutput(soClient, defaultDataOutputId, updateData, fromPreconfiguration) {
    const originalOutput = await this.get(soClient, defaultDataOutputId);
    this._validateFieldsAreEditable(originalOutput, updateData, defaultDataOutputId, fromPreconfiguration);
    _audit_logging.auditLoggingService.writeCustomSoAuditLog({
      action: 'update',
      id: outputIdToUuid(defaultDataOutputId),
      savedObjectType: _constants.OUTPUT_SAVED_OBJECT_TYPE
    });
    return await this.encryptedSoClient.update(SAVED_OBJECT_TYPE, outputIdToUuid(defaultDataOutputId), updateData);
  }
  _validateFieldsAreEditable(originalOutput, data, id, fromPreconfiguration) {
    if (originalOutput.is_preconfigured) {
      if (!fromPreconfiguration) {
        var _originalOutput$allow;
        const allowEditFields = (_originalOutput$allow = originalOutput.allow_edit) !== null && _originalOutput$allow !== void 0 ? _originalOutput$allow : [];
        const allKeys = Array.from(new Set([...Object.keys(data)]));
        for (const key of allKeys) {
          if ((!!originalOutput[key] || !!data[key]) && !allowEditFields.includes(key) && !(0, _fastDeepEqual.default)(originalOutput[key], data[key])) {
            throw new _errors.OutputUnauthorizedError(`Preconfigured output ${id} ${key} cannot be updated outside of kibana config file.`);
          }
        }
      }
    }
  }
  async ensureDefaultOutput(soClient, esClient) {
    const outputs = await this.list(soClient);
    const defaultOutput = outputs.items.find(o => o.is_default);
    const defaultMonitoringOutput = outputs.items.find(o => o.is_default_monitoring);
    if (!defaultOutput) {
      const newDefaultOutput = {
        ..._constants.DEFAULT_OUTPUT,
        hosts: this.getDefaultESHosts(),
        ca_sha256: _app_context.appContextService.getConfig().agents.elasticsearch.ca_sha256,
        is_default_monitoring: !defaultMonitoringOutput
      };
      return await this.create(soClient, esClient, newDefaultOutput, {
        id: _constants.DEFAULT_OUTPUT_ID,
        overwrite: true
      });
    }
    return defaultOutput;
  }
  getDefaultESHosts() {
    var _agents, _agents$elasticsearch, _agents$elasticsearch2;
    const cloud = _app_context.appContextService.getCloud();
    const cloudUrl = cloud === null || cloud === void 0 ? void 0 : cloud.elasticsearchUrl;
    const cloudHosts = cloudUrl ? [cloudUrl] : undefined;
    const flagHosts = (_agents = _app_context.appContextService.getConfig().agents) !== null && _agents !== void 0 && (_agents$elasticsearch = _agents.elasticsearch) !== null && _agents$elasticsearch !== void 0 && _agents$elasticsearch.hosts && (_agents$elasticsearch2 = _app_context.appContextService.getConfig().agents.elasticsearch.hosts) !== null && _agents$elasticsearch2 !== void 0 && _agents$elasticsearch2.length ? _app_context.appContextService.getConfig().agents.elasticsearch.hosts : undefined;
    return cloudHosts || flagHosts || DEFAULT_ES_HOSTS;
  }
  async getDefaultDataOutputId(soClient) {
    const outputs = await this._getDefaultDataOutputsSO();
    if (!outputs.saved_objects.length) {
      return null;
    }
    return outputSavedObjectToOutput(outputs.saved_objects[0]).id;
  }
  async getDefaultMonitoringOutputId(soClient) {
    const outputs = await this._getDefaultMonitoringOutputsSO(soClient);
    if (!outputs.saved_objects.length) {
      return null;
    }
    return outputSavedObjectToOutput(outputs.saved_objects[0]).id;
  }
  async create(soClient, esClient, output, options) {
    var _options$fromPreconfi;
    const logger = _app_context.appContextService.getLogger();
    logger.debug(`Creating new output`);
    const data = {
      ...(0, _lodash.omit)(output, ['ssl', 'secrets'])
    };
    if ((0, _output_helpers.outputTypeSupportPresets)(data.type)) {
      var _output$config_yaml;
      if (data.preset === 'balanced' && (0, _output_helpers.outputYmlIncludesReservedPerformanceKey)((_output$config_yaml = output.config_yaml) !== null && _output$config_yaml !== void 0 ? _output$config_yaml : '', _jsYaml.load)) {
        throw new _errors.OutputInvalidError(`preset cannot be balanced when config_yaml contains one of ${_constants2.RESERVED_CONFIG_YML_KEYS.join(', ')}`);
      }
    }
    const defaultDataOutputId = await this.getDefaultDataOutputId(soClient);
    if (output.type === _constants2.outputType.Logstash || output.type === _constants2.outputType.Kafka) {
      var _appContextService$ge;
      await validateLogstashOutputNotUsedInAPMPolicy(undefined, data.is_default);
      if (!((_appContextService$ge = _app_context.appContextService.getEncryptedSavedObjectsSetup()) !== null && _appContextService$ge !== void 0 && _appContextService$ge.canEncrypt)) {
        throw new _errors.FleetEncryptedSavedObjectEncryptionKeyRequired(`${output.type} output needs encrypted saved object api key to be set`);
      }
    }
    const {
      policiesWithFleetServer,
      policiesWithSynthetics
    } = await findPoliciesWithFleetServerOrSynthetics();
    const agentlessPolicies = await (0, _helpers.findAgentlessPolicies)();
    await updateAgentPoliciesDataOutputId(soClient, esClient, data, data.is_default, defaultDataOutputId, _lodash.default.uniq([...policiesWithFleetServer, ...policiesWithSynthetics, ...agentlessPolicies]), (_options$fromPreconfi = options === null || options === void 0 ? void 0 : options.fromPreconfiguration) !== null && _options$fromPreconfi !== void 0 ? _options$fromPreconfi : false);

    // ensure only default output exists
    if (data.is_default) {
      if (defaultDataOutputId && defaultDataOutputId !== (options === null || options === void 0 ? void 0 : options.id)) {
        var _options$fromPreconfi2;
        await this._updateDefaultOutput(soClient, defaultDataOutputId, {
          is_default: false
        }, (_options$fromPreconfi2 = options === null || options === void 0 ? void 0 : options.fromPreconfiguration) !== null && _options$fromPreconfi2 !== void 0 ? _options$fromPreconfi2 : false);
      }
    }
    if (data.is_default_monitoring) {
      const defaultMonitoringOutputId = await this.getDefaultMonitoringOutputId(soClient);
      if (defaultMonitoringOutputId && defaultMonitoringOutputId !== (options === null || options === void 0 ? void 0 : options.id)) {
        var _options$fromPreconfi3;
        await this._updateDefaultOutput(soClient, defaultMonitoringOutputId, {
          is_default_monitoring: false
        }, (_options$fromPreconfi3 = options === null || options === void 0 ? void 0 : options.fromPreconfiguration) !== null && _options$fromPreconfi3 !== void 0 ? _options$fromPreconfi3 : false);
      }
    }
    if ((data.type === _constants2.outputType.Elasticsearch || data.type === _constants2.outputType.RemoteElasticsearch) && data.hosts) {
      data.hosts = data.hosts.map(_services.normalizeHostsForAgents);
    }
    if (options !== null && options !== void 0 && options.id) {
      data.output_id = options === null || options === void 0 ? void 0 : options.id;
    }
    if (output.ssl) {
      data.ssl = JSON.stringify(output.ssl);
    }

    // Remove the shipper data if the shipper is not enabled from the yaml config
    if (!output.config_yaml && output.shipper) {
      data.shipper = null;
    }
    if (!data.preset && data.type === _constants2.outputType.Elasticsearch) {
      var _data$config_yaml;
      data.preset = (0, _output_helpers.getDefaultPresetForEsOutput)((_data$config_yaml = data.config_yaml) !== null && _data$config_yaml !== void 0 ? _data$config_yaml : '', _jsYaml.load);
    }
    if (output.config_yaml) {
      var _configJs$shipper;
      const configJs = (0, _jsYaml.load)(output.config_yaml);
      const isShipperDisabled = !(configJs !== null && configJs !== void 0 && configJs.shipper) || (configJs === null || configJs === void 0 ? void 0 : (_configJs$shipper = configJs.shipper) === null || _configJs$shipper === void 0 ? void 0 : _configJs$shipper.enabled) === false;
      if (isShipperDisabled && output.shipper) {
        data.shipper = null;
      }
    }
    if (output.type === _constants2.outputType.Kafka && data.type === _constants2.outputType.Kafka) {
      var _output$sasl, _output$random, _output$round_robin;
      if (!output.version) {
        data.version = '1.0.0';
      }
      if (!output.compression) {
        data.compression = _constants2.kafkaCompressionType.Gzip;
      }
      if (!output.compression || output.compression === _constants2.kafkaCompressionType.Gzip && !output.compression_level) {
        data.compression_level = 4;
      }
      if (!output.client_id) {
        data.client_id = 'Elastic';
      }
      if (output.username && output.password && !((_output$sasl = output.sasl) !== null && _output$sasl !== void 0 && _output$sasl.mechanism)) {
        data.sasl = {
          mechanism: _constants2.kafkaSaslMechanism.Plain
        };
      }
      if (!output.partition) {
        data.partition = _constants2.kafkaPartitionType.Hash;
      }
      if (output.partition === _constants2.kafkaPartitionType.Random && !((_output$random = output.random) !== null && _output$random !== void 0 && _output$random.group_events)) {
        data.random = {
          group_events: 1
        };
      }
      if (output.partition === _constants2.kafkaPartitionType.RoundRobin && !((_output$round_robin = output.round_robin) !== null && _output$round_robin !== void 0 && _output$round_robin.group_events)) {
        data.round_robin = {
          group_events: 1
        };
      }
      if (!output.timeout) {
        data.timeout = 30;
      }
      if (!output.broker_timeout) {
        data.broker_timeout = 10;
      }
      if (output.required_acks === null || output.required_acks === undefined) {
        // required_acks can be 0
        data.required_acks = _constants2.kafkaAcknowledgeReliabilityLevel.Commit;
      }
    }
    const id = options !== null && options !== void 0 && options.id ? outputIdToUuid(options.id) : _server.SavedObjectsUtils.generateId();

    // Store secret values if enabled; if not, store plain text values
    if (await (0, _secrets.isOutputSecretStorageEnabled)(esClient, soClient)) {
      const {
        output: outputWithSecrets
      } = await (0, _secrets.extractAndWriteOutputSecrets)({
        output,
        esClient,
        secretHashes: output.is_preconfigured ? options === null || options === void 0 ? void 0 : options.secretHashes : undefined
      });
      if (outputWithSecrets.secrets) data.secrets = outputWithSecrets.secrets;
    } else {
      if (output.type === _constants2.outputType.Logstash && data.type === _constants2.outputType.Logstash) {
        var _output$ssl, _output$secrets, _output$secrets$ssl;
        if (!((_output$ssl = output.ssl) !== null && _output$ssl !== void 0 && _output$ssl.key) && (_output$secrets = output.secrets) !== null && _output$secrets !== void 0 && (_output$secrets$ssl = _output$secrets.ssl) !== null && _output$secrets$ssl !== void 0 && _output$secrets$ssl.key) {
          data.ssl = JSON.stringify({
            ...output.ssl,
            ...output.secrets.ssl
          });
        }
      } else if (output.type === _constants2.outputType.Kafka && data.type === _constants2.outputType.Kafka) {
        var _output$secrets2, _output$ssl2, _output$secrets4, _output$secrets4$ssl;
        if (!output.password && (_output$secrets2 = output.secrets) !== null && _output$secrets2 !== void 0 && _output$secrets2.password) {
          var _output$secrets3;
          data.password = (_output$secrets3 = output.secrets) === null || _output$secrets3 === void 0 ? void 0 : _output$secrets3.password;
        }
        if (!((_output$ssl2 = output.ssl) !== null && _output$ssl2 !== void 0 && _output$ssl2.key) && (_output$secrets4 = output.secrets) !== null && _output$secrets4 !== void 0 && (_output$secrets4$ssl = _output$secrets4.ssl) !== null && _output$secrets4$ssl !== void 0 && _output$secrets4$ssl.key) {
          data.ssl = JSON.stringify({
            ...output.ssl,
            ...output.secrets.ssl
          });
        }
      } else if (output.type === _constants2.outputType.RemoteElasticsearch && data.type === _constants2.outputType.RemoteElasticsearch) {
        var _output$secrets5;
        if (!output.service_token && (_output$secrets5 = output.secrets) !== null && _output$secrets5 !== void 0 && _output$secrets5.service_token) {
          var _output$secrets6;
          data.service_token = (_output$secrets6 = output.secrets) === null || _output$secrets6 === void 0 ? void 0 : _output$secrets6.service_token;
        }
      }
    }
    _audit_logging.auditLoggingService.writeCustomSoAuditLog({
      action: 'create',
      id,
      savedObjectType: _constants.OUTPUT_SAVED_OBJECT_TYPE
    });
    const newSo = await this.encryptedSoClient.create(SAVED_OBJECT_TYPE, data, {
      overwrite: (options === null || options === void 0 ? void 0 : options.overwrite) || (options === null || options === void 0 ? void 0 : options.fromPreconfiguration),
      id
    });
    logger.debug(`Created new output ${id}`);
    return outputSavedObjectToOutput(newSo);
  }
  async bulkGet(soClient, ids, {
    ignoreNotFound = false
  } = {
    ignoreNotFound: true
  }) {
    const res = await this.encryptedSoClient.bulkGet(ids.map(id => ({
      id: outputIdToUuid(id),
      type: SAVED_OBJECT_TYPE
    })));
    return res.saved_objects.map(so => {
      if (so.error) {
        if (!ignoreNotFound || so.error.statusCode !== 404) {
          throw so.error;
        }
        return undefined;
      }
      return outputSavedObjectToOutput(so);
    }).filter(output => typeof output !== 'undefined');
  }
  async list(soClient) {
    const outputs = await this.encryptedSoClient.find({
      type: SAVED_OBJECT_TYPE,
      page: 1,
      perPage: _constants2.SO_SEARCH_LIMIT,
      sortField: 'is_default',
      sortOrder: 'desc'
    });
    for (const output of outputs.saved_objects) {
      _audit_logging.auditLoggingService.writeCustomSoAuditLog({
        action: 'get',
        id: output.id,
        savedObjectType: _constants.OUTPUT_SAVED_OBJECT_TYPE
      });
    }
    return {
      items: outputs.saved_objects.map(outputSavedObjectToOutput),
      total: outputs.total,
      page: outputs.page,
      perPage: outputs.per_page
    };
  }
  async listAllForProxyId(soClient, proxyId) {
    const outputs = await this.encryptedSoClient.find({
      type: SAVED_OBJECT_TYPE,
      page: 1,
      perPage: _constants2.SO_SEARCH_LIMIT,
      searchFields: ['proxy_id'],
      search: (0, _saved_object.escapeSearchQueryPhrase)(proxyId)
    });
    for (const output of outputs.saved_objects) {
      _audit_logging.auditLoggingService.writeCustomSoAuditLog({
        action: 'get',
        id: output.id,
        savedObjectType: _constants.OUTPUT_SAVED_OBJECT_TYPE
      });
    }
    return {
      items: outputs.saved_objects.map(outputSavedObjectToOutput),
      total: outputs.total,
      page: outputs.page,
      perPage: outputs.per_page
    };
  }
  async get(soClient, id) {
    const outputSO = await this.encryptedSoClient.get(SAVED_OBJECT_TYPE, outputIdToUuid(id));
    _audit_logging.auditLoggingService.writeCustomSoAuditLog({
      action: 'get',
      id: outputSO.id,
      savedObjectType: _constants.OUTPUT_SAVED_OBJECT_TYPE
    });
    if (outputSO.error) {
      throw new _errors.FleetError(outputSO.error.message);
    }
    return outputSavedObjectToOutput(outputSO);
  }
  async delete(soClient, id, {
    fromPreconfiguration = false
  } = {
    fromPreconfiguration: false
  }) {
    const logger = _app_context.appContextService.getLogger();
    logger.debug(`Deleting output ${id}`);
    const originalOutput = await this.get(soClient, id);
    if (originalOutput.is_preconfigured && !fromPreconfiguration) {
      throw new _errors.OutputUnauthorizedError(`Preconfigured output ${id} cannot be deleted outside of kibana config file.`);
    }
    if (originalOutput.is_default && !fromPreconfiguration) {
      throw new _errors.OutputUnauthorizedError(`Default output ${id} cannot be deleted.`);
    }
    if (originalOutput.is_default_monitoring && !fromPreconfiguration) {
      throw new _errors.OutputUnauthorizedError(`Default monitoring output ${id} cannot be deleted.`);
    }
    await _package_policy.packagePolicyService.removeOutputFromAll(_app_context.appContextService.getInternalUserESClient(), id, {
      force: fromPreconfiguration
    });
    await _agent_policy.agentPolicyService.removeOutputFromAll(_app_context.appContextService.getInternalUserESClient(), id, {
      force: fromPreconfiguration
    });
    _audit_logging.auditLoggingService.writeCustomSoAuditLog({
      action: 'delete',
      id: outputIdToUuid(id),
      savedObjectType: _constants.OUTPUT_SAVED_OBJECT_TYPE
    });
    const soDeleteResult = this.encryptedSoClient.delete(SAVED_OBJECT_TYPE, outputIdToUuid(id));
    await (0, _secrets.deleteOutputSecrets)({
      esClient: _app_context.appContextService.getInternalUserESClient(),
      output: originalOutput
    });
    logger.debug(`Deleted output ${id}`);
    return soDeleteResult;
  }
  async update(soClient, esClient, id, data, {
    fromPreconfiguration = false,
    secretHashes
  } = {
    fromPreconfiguration: false
  }) {
    var _data$type, _data$is_default2;
    const logger = _app_context.appContextService.getLogger();
    logger.debug(`Updating output ${id}`);
    let secretsToDelete = [];
    const originalOutput = await this.get(soClient, id);
    this._validateFieldsAreEditable(originalOutput, data, id, fromPreconfiguration);
    if (originalOutput.is_default && data.is_default === false || data.is_default_monitoring === false && originalOutput.is_default_monitoring) {
      throw new _errors.OutputUnauthorizedError(`Default output ${id} cannot be set to is_default=false or is_default_monitoring=false manually. Make another output the default first.`);
    }
    const updateData = {
      ...(0, _lodash.omit)(data, ['ssl', 'secrets'])
    };
    if (updateData.type && (0, _output_helpers.outputTypeSupportPresets)(updateData.type)) {
      var _updateData$config_ya;
      if (updateData.preset === 'balanced' && (0, _output_helpers.outputYmlIncludesReservedPerformanceKey)((_updateData$config_ya = updateData.config_yaml) !== null && _updateData$config_ya !== void 0 ? _updateData$config_ya : '', _jsYaml.load)) {
        throw new _errors.OutputInvalidError(`preset cannot be balanced when config_yaml contains one of ${_constants2.RESERVED_CONFIG_YML_KEYS.join(', ')}`);
      }
    }
    const mergedType = (_data$type = data.type) !== null && _data$type !== void 0 ? _data$type : originalOutput.type;
    const mergedIsDefault = (_data$is_default2 = data.is_default) !== null && _data$is_default2 !== void 0 ? _data$is_default2 : originalOutput.is_default;
    const defaultDataOutputId = await this.getDefaultDataOutputId(soClient);
    if (mergedType !== originalOutput.type || originalOutput.is_default !== mergedIsDefault) {
      await validateTypeChanges(esClient, id, updateData, originalOutput, defaultDataOutputId, fromPreconfiguration);
    }
    const removeKafkaFields = target => {
      target.version = null;
      target.key = null;
      target.compression = null;
      target.compression_level = null;
      target.connection_type = null;
      target.client_id = null;
      target.auth_type = null;
      target.username = null;
      target.password = null;
      target.sasl = null;
      target.partition = null;
      target.random = null;
      target.round_robin = null;
      target.hash = null;
      target.topics = null;
      target.topic = null;
      target.headers = null;
      target.timeout = null;
      target.broker_timeout = null;
      target.required_acks = null;
      target.ssl = null;
    };

    // If the output type changed
    if (data.type && data.type !== originalOutput.type) {
      if (data.type === _constants2.outputType.Elasticsearch && updateData.type === _constants2.outputType.Elasticsearch) {
        updateData.preset = null;
      }
      if (data.type !== _constants2.outputType.Kafka && originalOutput.type === _constants2.outputType.Kafka) {
        removeKafkaFields(updateData);
      }
      if (data.type === _constants2.outputType.Logstash) {
        // remove ES specific field
        updateData.ca_trusted_fingerprint = null;
        updateData.ca_sha256 = null;
        delete updateData.service_token;
      }
      if (data.type !== _constants2.outputType.Logstash) {
        // remove logstash specific field
        updateData.ssl = null;
      }
      if (data.type === _constants2.outputType.Kafka && updateData.type === _constants2.outputType.Kafka) {
        var _data$sasl, _data$random, _data$round_robin;
        updateData.ca_trusted_fingerprint = null;
        updateData.ca_sha256 = null;
        if (!data.version) {
          updateData.version = '1.0.0';
        }
        if (!data.compression) {
          updateData.compression = _constants2.kafkaCompressionType.Gzip;
        }
        if (!data.compression || data.compression === _constants2.kafkaCompressionType.Gzip && !data.compression_level) {
          updateData.compression_level = 4;
        }
        if (data.compression && data.compression !== _constants2.kafkaCompressionType.Gzip) {
          // Clear compression level if compression is not gzip
          updateData.compression_level = null;
        }
        if (!data.client_id) {
          updateData.client_id = 'Elastic';
        }
        if (data.username && data.password && !((_data$sasl = data.sasl) !== null && _data$sasl !== void 0 && _data$sasl.mechanism)) {
          updateData.sasl = {
            mechanism: _constants2.kafkaSaslMechanism.Plain
          };
        }
        if (!data.partition) {
          updateData.partition = _constants2.kafkaPartitionType.Hash;
        }
        if (data.partition === _constants2.kafkaPartitionType.Random && !((_data$random = data.random) !== null && _data$random !== void 0 && _data$random.group_events)) {
          updateData.random = {
            group_events: 1
          };
        }
        if (data.partition === _constants2.kafkaPartitionType.RoundRobin && !((_data$round_robin = data.round_robin) !== null && _data$round_robin !== void 0 && _data$round_robin.group_events)) {
          updateData.round_robin = {
            group_events: 1
          };
        }
        if (!data.timeout) {
          updateData.timeout = 30;
        }
        if (!data.broker_timeout) {
          updateData.broker_timeout = 10;
        }
        if (updateData.required_acks === null || updateData.required_acks === undefined) {
          // required_acks can be 0
          updateData.required_acks = _constants2.kafkaAcknowledgeReliabilityLevel.Commit;
        }
      }
    }
    if (data.ssl) {
      updateData.ssl = JSON.stringify(data.ssl);
    } else if (data.ssl === null) {
      // Explicitly set to null to allow to delete the field
      updateData.ssl = null;
    }
    if (data.type === _constants2.outputType.Kafka && updateData.type === _constants2.outputType.Kafka) {
      if (!data.password) {
        updateData.password = null;
      }
      if (!data.username) {
        updateData.username = null;
      }
      if (!data.ssl) {
        updateData.ssl = null;
      }
      if (!data.sasl) {
        updateData.sasl = null;
      }
    }

    // ensure only default output exists
    if (data.is_default) {
      if (defaultDataOutputId && defaultDataOutputId !== id) {
        await this._updateDefaultOutput(soClient, defaultDataOutputId, {
          is_default: false
        }, fromPreconfiguration);
      }
    }
    if (data.is_default_monitoring) {
      const defaultMonitoringOutputId = await this.getDefaultMonitoringOutputId(soClient);
      if (defaultMonitoringOutputId && defaultMonitoringOutputId !== id) {
        await this._updateDefaultOutput(soClient, defaultMonitoringOutputId, {
          is_default_monitoring: false
        }, fromPreconfiguration);
      }
    }
    if ((mergedType === _constants2.outputType.Elasticsearch || mergedType === _constants2.outputType.RemoteElasticsearch) && updateData.hosts) {
      updateData.hosts = updateData.hosts.map(_services.normalizeHostsForAgents);
    }
    if (data.type === _constants2.outputType.RemoteElasticsearch && updateData.type === _constants2.outputType.RemoteElasticsearch) {
      if (!data.service_token) {
        updateData.service_token = null;
      }
    }
    if (!data.preset && data.type === _constants2.outputType.Elasticsearch) {
      var _data$config_yaml2;
      updateData.preset = (0, _output_helpers.getDefaultPresetForEsOutput)((_data$config_yaml2 = data.config_yaml) !== null && _data$config_yaml2 !== void 0 ? _data$config_yaml2 : '', _jsYaml.load);
    }

    // Remove the shipper data if the shipper is not enabled from the yaml config
    if (!data.config_yaml && data.shipper) {
      updateData.shipper = null;
    }
    if (data.config_yaml) {
      var _configJs$shipper2;
      const configJs = (0, _jsYaml.load)(data.config_yaml);
      const isShipperDisabled = !(configJs !== null && configJs !== void 0 && configJs.shipper) || (configJs === null || configJs === void 0 ? void 0 : (_configJs$shipper2 = configJs.shipper) === null || _configJs$shipper2 === void 0 ? void 0 : _configJs$shipper2.enabled) === false;
      if (isShipperDisabled && data.shipper) {
        updateData.shipper = null;
      }
    }

    // Store secret values if enabled; if not, store plain text values
    if (await (0, _secrets.isOutputSecretStorageEnabled)(esClient, soClient)) {
      const secretsRes = await (0, _secrets.extractAndUpdateOutputSecrets)({
        oldOutput: originalOutput,
        outputUpdate: data,
        esClient,
        secretHashes: data.is_preconfigured ? secretHashes : undefined
      });
      updateData.secrets = secretsRes.outputUpdate.secrets;
      secretsToDelete = secretsRes.secretsToDelete;
    } else {
      if (data.type === _constants2.outputType.Logstash && updateData.type === _constants2.outputType.Logstash) {
        var _data$ssl, _data$secrets, _data$secrets$ssl;
        if (!((_data$ssl = data.ssl) !== null && _data$ssl !== void 0 && _data$ssl.key) && (_data$secrets = data.secrets) !== null && _data$secrets !== void 0 && (_data$secrets$ssl = _data$secrets.ssl) !== null && _data$secrets$ssl !== void 0 && _data$secrets$ssl.key) {
          updateData.ssl = JSON.stringify({
            ...data.ssl,
            ...data.secrets.ssl
          });
        }
      } else if (data.type === _constants2.outputType.Kafka && updateData.type === _constants2.outputType.Kafka) {
        var _data$secrets2, _data$ssl2, _data$secrets4, _data$secrets4$ssl;
        if (!data.password && (_data$secrets2 = data.secrets) !== null && _data$secrets2 !== void 0 && _data$secrets2.password) {
          var _data$secrets3;
          updateData.password = (_data$secrets3 = data.secrets) === null || _data$secrets3 === void 0 ? void 0 : _data$secrets3.password;
        }
        if (!((_data$ssl2 = data.ssl) !== null && _data$ssl2 !== void 0 && _data$ssl2.key) && (_data$secrets4 = data.secrets) !== null && _data$secrets4 !== void 0 && (_data$secrets4$ssl = _data$secrets4.ssl) !== null && _data$secrets4$ssl !== void 0 && _data$secrets4$ssl.key) {
          updateData.ssl = JSON.stringify({
            ...data.ssl,
            ...data.secrets.ssl
          });
        }
      } else if (data.type === _constants2.outputType.RemoteElasticsearch && updateData.type === _constants2.outputType.RemoteElasticsearch) {
        var _data$secrets5;
        if (!data.service_token && (_data$secrets5 = data.secrets) !== null && _data$secrets5 !== void 0 && _data$secrets5.service_token) {
          var _data$secrets6;
          updateData.service_token = (_data$secrets6 = data.secrets) === null || _data$secrets6 === void 0 ? void 0 : _data$secrets6.service_token;
        }
      }
    }
    (0, _so_helpers.patchUpdateDataWithRequireEncryptedAADFields)(updateData, originalOutput);
    _audit_logging.auditLoggingService.writeCustomSoAuditLog({
      action: 'update',
      id: outputIdToUuid(id),
      savedObjectType: _constants.OUTPUT_SAVED_OBJECT_TYPE
    });
    const outputSO = await this.encryptedSoClient.update(SAVED_OBJECT_TYPE, outputIdToUuid(id), updateData);
    if (outputSO.error) {
      throw new _errors.FleetError(outputSO.error.message);
    }
    if (secretsToDelete.length) {
      try {
        await (0, _secrets.deleteSecrets)({
          esClient,
          ids: secretsToDelete.map(s => s.id)
        });
      } catch (err) {
        logger.warn(`Error cleaning up secrets for output ${id}: ${err.message}`);
      }
    }
    logger.debug(`Updated output ${id}`);
  }
  async backfillAllOutputPresets(soClient, esClient) {
    const outputs = await this.list(soClient);
    await (0, _pMap.default)(outputs.items.filter(output => (0, _output_helpers.outputTypeSupportPresets)(output.type) && !output.preset), async output => {
      var _output$config_yaml2;
      const preset = (0, _output_helpers.getDefaultPresetForEsOutput)((_output$config_yaml2 = output.config_yaml) !== null && _output$config_yaml2 !== void 0 ? _output$config_yaml2 : '', _jsYaml.load);
      await outputService.update(soClient, esClient, output.id, {
        preset
      }, {
        fromPreconfiguration: true
      });
      await _agent_policy.agentPolicyService.bumpAllAgentPoliciesForOutput(esClient, output.id);
    }, {
      concurrency: 5
    });
  }
  async getLatestOutputHealth(esClient, id) {
    var _latestHit$message;
    const lastUpdateTime = await this.getOutputLastUpdateTime(id);
    const mustFilter = [];
    if (lastUpdateTime) {
      mustFilter.push({
        range: {
          '@timestamp': {
            gte: lastUpdateTime
          }
        }
      });
    }
    const response = await esClient.search({
      index: _constants.OUTPUT_HEALTH_DATA_STREAM,
      query: {
        bool: {
          filter: {
            term: {
              output: id
            }
          },
          must: mustFilter
        }
      },
      sort: {
        '@timestamp': 'desc'
      },
      size: 1
    }, {
      ignore: [404]
    });
    if (!response.hits || response.hits.hits.length === 0) {
      return {
        state: 'UNKNOWN',
        message: '',
        timestamp: ''
      };
    }
    const latestHit = response.hits.hits[0]._source;
    return {
      state: latestHit.state,
      message: (_latestHit$message = latestHit.message) !== null && _latestHit$message !== void 0 ? _latestHit$message : '',
      timestamp: latestHit['@timestamp']
    };
  }
  async getOutputLastUpdateTime(id) {
    const outputSO = await this.encryptedSoClient.get(SAVED_OBJECT_TYPE, outputIdToUuid(id));
    if (outputSO.error) {
      _app_context.appContextService.getLogger().debug(`Error getting output ${id} SO, using updated_at:undefined, cause: ${outputSO.error.message}`);
      return undefined;
    }
    return outputSO.updated_at;
  }
}
const outputService = exports.outputService = new OutputService();