"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fileUploadRoutes = fileUploadRoutes;
var _configSchema = require("@kbn/config-schema");
var _constants = require("@kbn/file-upload-common/src/constants");
var _error_wrapper = require("./error_wrapper");
var _import_data = require("./import_data");
var _get_time_field_range = require("./get_time_field_range");
var _analyze_file = require("./analyze_file");
var _telemetry = require("./telemetry");
var _schemas = require("./schemas");
var _check_privileges = require("./check_privileges");
var _preview_index_time_range = require("./preview_index_time_range");
var _preview_tika_contents = require("./preview_tika_contents");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Routes for the file upload.
 */
function fileUploadRoutes(coreSetup, logger) {
  const router = coreSetup.http.createRouter();
  router.versioned.get({
    path: '/internal/file_upload/has_import_permission',
    access: 'internal'
  }).addVersion({
    version: '1',
    security: {
      authz: {
        enabled: false,
        reason: 'This route is opted out from authorization because permissions will be checked by elasticsearch'
      }
    },
    validate: {
      request: {
        query: _configSchema.schema.object({
          indexName: _configSchema.schema.maybe(_configSchema.schema.string()),
          checkCreateDataView: _configSchema.schema.boolean(),
          checkHasManagePipeline: _configSchema.schema.boolean()
        })
      }
    }
  }, async (context, request, response) => {
    try {
      var _pluginsStart$securit;
      const [, pluginsStart] = await coreSetup.getStartServices();
      const {
        indexName,
        checkCreateDataView,
        checkHasManagePipeline
      } = request.query;
      const {
        hasImportPermission
      } = await (0, _check_privileges.checkFileUploadPrivileges)({
        authorization: (_pluginsStart$securit = pluginsStart.security) === null || _pluginsStart$securit === void 0 ? void 0 : _pluginsStart$securit.authz,
        request,
        indexName,
        checkCreateDataView,
        checkHasManagePipeline
      });
      return response.ok({
        body: {
          hasImportPermission
        }
      });
    } catch (e) {
      logger.warn(`Unable to check import permission, error: ${e.message}`);
      return response.ok({
        body: {
          hasImportPermission: false
        }
      });
    }
  });

  /**
   * @apiGroup FileDataVisualizer
   *
   * @api {post} /internal/file_upload/analyze_file Analyze file data
   * @apiName AnalyzeFile
   * @apiDescription Performs analysis of the file data.
   *
   * @apiSchema (query) analyzeFileQuerySchema
   */
  router.versioned.post({
    path: '/internal/file_upload/analyze_file',
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['fileUpload:analyzeFile']
      }
    },
    options: {
      body: {
        accepts: ['text/*', 'application/json'],
        maxBytes: _constants.MAX_FILE_SIZE_BYTES
      }
    }
  }).addVersion({
    version: '1',
    validate: {
      request: {
        body: _configSchema.schema.any(),
        query: _schemas.analyzeFileQuerySchema
      }
    }
  }, async (context, request, response) => {
    try {
      const esClient = (await context.core).elasticsearch.client;
      const result = await (0, _analyze_file.analyzeFile)(esClient, request.body, request.query);
      return response.ok({
        body: result
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  });

  /**
   * @apiGroup FileDataVisualizer
   *
   * @api {post} /internal/file_upload/initialize_import Initialize import file process
   * @apiName InitializeImportFile
   * @apiDescription Creates an index and ingest pipelines for importing file data.
   *
   * @apiSchema (body) initializeImportFileBodySchema
   */
  router.versioned.post({
    path: '/internal/file_upload/initialize_import',
    access: 'internal',
    options: {
      body: {
        accepts: ['application/json'],
        maxBytes: _constants.MAX_FILE_SIZE_BYTES
      }
    }
  }).addVersion({
    version: '1',
    security: {
      authz: {
        enabled: false,
        reason: 'This route is opted out from authorization because permissions will be checked by elasticsearch'
      }
    },
    validate: {
      request: {
        body: _schemas.initializeImportFileBodySchema
      }
    }
  }, async (context, request, response) => {
    try {
      const {
        index,
        settings,
        mappings,
        ingestPipelines,
        existingIndex
      } = request.body;
      const esClient = (await context.core).elasticsearch.client;
      await (0, _telemetry.updateTelemetry)();
      const {
        initializeImport
      } = (0, _import_data.importDataProvider)(esClient);
      const result = await initializeImport(index, settings, mappings, ingestPipelines, existingIndex);
      return response.ok({
        body: result
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  });

  /**
   * @apiGroup FileDataVisualizer
   *
   * @api {post} /internal/file_upload/import Import file data
   * @apiName ImportFile
   * @apiDescription Imports file data into elasticsearch index.
   *
   * @apiSchema (body) importFileBodySchema
   */
  router.versioned.post({
    path: '/internal/file_upload/import',
    access: 'internal',
    options: {
      body: {
        accepts: ['application/json'],
        maxBytes: _constants.MAX_FILE_SIZE_BYTES
      }
    }
  }).addVersion({
    version: '1',
    security: {
      authz: {
        enabled: false,
        reason: 'This route is opted out from authorization because permissions will be checked by elasticsearch'
      }
    },
    validate: {
      request: {
        query: _configSchema.schema.object({
          id: _configSchema.schema.maybe(_configSchema.schema.string())
        }),
        body: _configSchema.schema.object({
          index: _configSchema.schema.string(),
          data: _configSchema.schema.arrayOf(_configSchema.schema.any()),
          settings: _configSchema.schema.maybe(_configSchema.schema.any()),
          /** Mappings */
          mappings: _configSchema.schema.any(),
          /** Ingest pipeline definition */
          ingestPipeline: _configSchema.schema.maybe(_configSchema.schema.object({
            id: _configSchema.schema.maybe(_configSchema.schema.string()),
            pipeline: _configSchema.schema.maybe(_configSchema.schema.any())
          })),
          createPipelines: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.maybe(_configSchema.schema.object({
            id: _configSchema.schema.maybe(_configSchema.schema.string()),
            pipeline: _configSchema.schema.maybe(_configSchema.schema.any())
          }))))
        })
      }
    }
  }, async (context, request, response) => {
    try {
      var _ingestPipeline$id;
      const {
        id
      } = request.query;
      const {
        index,
        data,
        settings,
        mappings,
        ingestPipeline,
        createPipelines
      } = request.body;
      const esClient = (await context.core).elasticsearch.client;
      const {
        initializeImport,
        importData
      } = (0, _import_data.importDataProvider)(esClient);
      if (id === undefined) {
        const pipelines = [...(ingestPipeline ? [ingestPipeline] : []), ...(createPipelines !== null && createPipelines !== void 0 ? createPipelines : [])];
        const result = await initializeImport(index, settings, mappings, pipelines);
        // format the response to match v1 response
        const body = {
          id: 'tempId',
          index: result.index,
          pipelineId: result.pipelineIds[0],
          success: result.success
        };
        return response.ok({
          body
        });
      }
      const result = await importData(index, (_ingestPipeline$id = ingestPipeline === null || ingestPipeline === void 0 ? void 0 : ingestPipeline.id) !== null && _ingestPipeline$id !== void 0 ? _ingestPipeline$id : '', data);
      return response.ok({
        body: result
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }).addVersion({
    version: '2',
    security: {
      authz: {
        enabled: false,
        reason: 'This route is opted out from authorization because permissions will be checked by elasticsearch'
      }
    },
    validate: {
      request: {
        body: _schemas.importFileBodySchema
      }
    }
  }, async (context, request, response) => {
    try {
      const {
        index,
        data,
        ingestPipelineId
      } = request.body;
      const esClient = (await context.core).elasticsearch.client;
      const {
        importData
      } = (0, _import_data.importDataProvider)(esClient);
      const result = await importData(index, ingestPipelineId, data);
      return response.ok({
        body: result
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  });

  /**
   * @apiGroup FileDataVisualizer
   *
   * @api {post} /internal/file_upload/index_exists ES indices exists wrapper checks if index exists
   * @apiName IndexExists
   */
  router.versioned.post({
    path: '/internal/file_upload/index_exists',
    access: 'internal'
  }).addVersion({
    version: '1',
    security: {
      authz: {
        enabled: false,
        reason: 'This route is opted out from authorization because permissions will be checked by elasticsearch'
      }
    },
    validate: {
      request: {
        body: _configSchema.schema.object({
          index: _configSchema.schema.string()
        })
      }
    }
  }, async (context, request, response) => {
    try {
      const esClient = (await context.core).elasticsearch.client;
      const indexExists = await esClient.asCurrentUser.indices.exists(request.body);
      return response.ok({
        body: {
          exists: indexExists
        }
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  });

  /**
   * @apiGroup FileDataVisualizer
   *
   * @api {post} /internal/file_upload/time_field_range Get time field range
   * @apiName GetTimeFieldRange
   * @apiDescription Returns the time range for the given index and query using the specified time range.
   *
   * @apiSchema (body) getTimeFieldRangeSchema
   *
   * @apiSuccess {Object} start start of time range with epoch and string properties.
   * @apiSuccess {Object} end end of time range with epoch and string properties.
   */
  router.versioned.post({
    path: '/internal/file_upload/time_field_range',
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['fileUpload:analyzeFile']
      }
    }
  }).addVersion({
    version: '1',
    validate: {
      request: {
        body: _configSchema.schema.object({
          /** Index or indexes for which to return the time range. */
          index: _configSchema.schema.oneOf([_configSchema.schema.string(), _configSchema.schema.arrayOf(_configSchema.schema.string())]),
          /** Name of the time field in the index. */
          timeFieldName: _configSchema.schema.string(),
          /** Query to match documents in the index(es). */
          query: _configSchema.schema.maybe(_configSchema.schema.any()),
          runtimeMappings: _configSchema.schema.maybe(_schemas.runtimeMappingsSchema)
        })
      }
    }
  }, async (context, request, response) => {
    try {
      const {
        index,
        timeFieldName,
        query,
        runtimeMappings
      } = request.body;
      const esClient = (await context.core).elasticsearch.client;
      const resp = await (0, _get_time_field_range.getTimeFieldRange)(esClient, index, timeFieldName, query, runtimeMappings);
      return response.ok({
        body: resp
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  });

  /**
   * @apiGroup FileDataVisualizer
   *
   * @api {post} /internal/file_upload/preview_index_time_range Predict the time range for an index using example documents
   * @apiName PreviewIndexTimeRange
   * @apiDescription Predict the time range for an index using example documents
   */
  router.versioned.post({
    path: '/internal/file_upload/preview_index_time_range',
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['fileUpload:analyzeFile']
      }
    }
  }).addVersion({
    version: '1',
    validate: {
      request: {
        body: _configSchema.schema.object({
          docs: _configSchema.schema.arrayOf(_configSchema.schema.any()),
          pipeline: _configSchema.schema.any(),
          timeField: _configSchema.schema.string()
        })
      }
    }
  }, async (context, request, response) => {
    try {
      const {
        docs,
        pipeline,
        timeField
      } = request.body;
      const esClient = (await context.core).elasticsearch.client;
      const resp = await (0, _preview_index_time_range.previewIndexTimeRange)(esClient, timeField, pipeline, docs);
      return response.ok({
        body: resp
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  });

  /**
   * @apiGroup FileDataVisualizer
   *
   * @api {post} /internal/file_upload/preview_tika_contents Returns the contents of a file using the attachment ingest processor
   * @apiName PreviewTikaContents
   * @apiDescription Preview the contents of a file using the attachment ingest processor
   */
  router.versioned.post({
    path: '/internal/file_upload/preview_tika_contents',
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['fileUpload:analyzeFile']
      }
    },
    options: {
      body: {
        accepts: ['application/json'],
        maxBytes: _constants.MAX_TIKA_FILE_SIZE_BYTES
      }
    }
  }).addVersion({
    version: '1',
    validate: {
      request: {
        body: _configSchema.schema.object({
          base64File: _configSchema.schema.string()
        })
      }
    }
  }, async (context, request, response) => {
    try {
      const {
        base64File
      } = request.body;
      const esClient = (await context.core).elasticsearch.client;
      const resp = await (0, _preview_tika_contents.previewTikaContents)(esClient, base64File);
      return response.ok({
        body: resp
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  });

  /**
   * @apiGroup FileDataVisualizer
   *
   * @api {post} /internal/file_upload/remove_pipelines Remove a list of ingest pipelines
   * @apiName RemovePipelines
   * @apiDescription Remove a list of ingest pipelines by id
   */
  router.versioned.delete({
    path: '/internal/file_upload/remove_pipelines/{pipelineIds}',
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['fileUpload:analyzeFile']
      }
    }
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _configSchema.schema.object({
          pipelineIds: _configSchema.schema.string()
        })
      }
    }
  }, async (context, request, response) => {
    try {
      const {
        pipelineIds
      } = request.params;
      const esClient = (await context.core).elasticsearch.client;
      const resp = await Promise.all(pipelineIds.split(',').map(id => esClient.asCurrentUser.ingest.deletePipeline({
        id
      })));
      return response.ok({
        body: resp
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  });
}