"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildQueryForFieldsForStringSources = buildQueryForFieldsForStringSources;
exports.buildQueryForFieldsFromSource = buildQueryForFieldsFromSource;
exports.buildQueryForFieldsInPolicies = buildQueryForFieldsInPolicies;
exports.collapseWrongArgumentTypeMessages = collapseWrongArgumentTypeMessages;
exports.getEnrichCommands = void 0;
exports.getMaxMinNumberOfParams = getMaxMinNumberOfParams;
var _esqlAst = require("@kbn/esql-ast");
var _helpers = require("../shared/helpers");
var _errors = require("./errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function buildQueryForFieldsFromSource(queryString, ast) {
  const firstCommand = ast[0];
  if (!firstCommand) return '';
  const sources = [];
  const metadataFields = [];
  if (firstCommand.name === 'ts') {
    const timeseries = firstCommand;
    sources.push(...timeseries.sources);
  } else if (firstCommand.name === 'from') {
    const fromSources = _esqlAst.mutate.commands.from.sources.list(firstCommand);
    const fromMetadataColumns = [..._esqlAst.mutate.commands.from.metadata.list(firstCommand)].map(([column]) => column);
    sources.push(...fromSources);
    if (fromMetadataColumns.length) metadataFields.push(...fromMetadataColumns);
  }
  const joinSummary = _esqlAst.mutate.commands.join.summarize({
    type: 'query',
    commands: ast
  });
  const joinIndices = joinSummary.map(({
    target: {
      index
    }
  }) => index);
  if (joinIndices.length > 0) {
    sources.push(...joinIndices);
  }
  if (sources.length === 0) {
    return queryString.substring(0, firstCommand.location.max + 1);
  }
  const from = metadataFields.length > 0 ? _esqlAst.synth.cmd`FROM ${sources} METADATA ${metadataFields}` : _esqlAst.synth.cmd`FROM ${sources}`;
  return from.toString();
}
function buildQueryForFieldsInPolicies(policies) {
  return `from ${policies.flatMap(({
    sourceIndices
  }) => sourceIndices).join(', ')} | keep ${policies.flatMap(({
    enrichFields
  }) => enrichFields).join(', ')}`;
}
function buildQueryForFieldsForStringSources(queryString, ast) {
  // filter out the query until the last GROK or DISSECT command
  const lastCommandIndex = ast.length - [...ast].reverse().findIndex(({
    name
  }) => ['grok', 'dissect'].includes(name));
  // we're sure it's not -1 because we check the commands chain before calling this function
  const nextCommandIndex = Math.min(lastCommandIndex + 1, ast.length - 1);
  const customQuery = queryString.substring(0, ast[nextCommandIndex].location.min).trimEnd();
  if (customQuery[customQuery.length - 1] === '|') {
    return customQuery.substring(0, customQuery.length - 1);
  }
  return customQuery;
}

/**
 * Returns the maximum and minimum number of parameters allowed by a function
 *
 * Used for too-many, too-few arguments validation
 */
function getMaxMinNumberOfParams(definition) {
  if (definition.signatures.length === 0) {
    return {
      min: 0,
      max: 0
    };
  }
  let min = Infinity;
  let max = 0;
  definition.signatures.forEach(({
    params,
    minParams
  }) => {
    min = Math.min(min, params.filter(({
      optional
    }) => !optional).length);
    max = Math.max(max, minParams ? Infinity : params.length);
  });
  return {
    min,
    max
  };
}

/**
 * We only want to report one message when any number of the elements in an array argument is of the wrong type
 */
function collapseWrongArgumentTypeMessages(messages, arg, funcName, argType, parentCommand, references) {
  if (!messages.some(({
    code
  }) => code === 'wrongArgumentType')) {
    return messages;
  }

  // Replace the individual "wrong argument type" messages with a single one for the whole array
  messages = messages.filter(({
    code
  }) => code !== 'wrongArgumentType');
  messages.push((0, _errors.getMessageFromId)({
    messageId: 'wrongArgumentType',
    values: {
      name: funcName,
      argType,
      value: `(${(0, _helpers.getAllArrayValues)(arg).join(', ')})`,
      givenType: `(${(0, _helpers.getAllArrayTypes)(arg, parentCommand, references).join(', ')})`
    },
    locations: {
      min: arg[0].location.min,
      max: arg[arg.length - 1].location.max
    }
  }));
  return messages;
}

/**
 * Collects all 'enrich' commands from a list of ESQL commands.
 * @param commands - The list of ESQL commands to search through.
 * This function traverses the provided ESQL commands and collects all commands with the name 'enrich'.
 * @returns {ESQLCommand[]} - An array of ESQLCommand objects that represent the 'enrich' commands found in the input.
 */
const getEnrichCommands = commands => _esqlAst.Walker.matchAll(commands, {
  type: 'command',
  name: 'enrich'
});
exports.getEnrichCommands = getEnrichCommands;