"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ESQLAstBuilderListener = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _completion = require("./factories/completion");
var _factories = require("./factories");
var _change_point = require("./factories/change_point");
var _dissect = require("./factories/dissect");
var _eval = require("./factories/eval");
var _fork = require("./factories/fork");
var _from = require("./factories/from");
var _grok = require("./factories/grok");
var _join = require("./factories/join");
var _limit = require("./factories/limit");
var _row = require("./factories/row");
var _sort = require("./factories/sort");
var _stats = require("./factories/stats");
var _where = require("./factories/where");
var _mv_expand = require("./factories/mv_expand");
var _keep = require("./factories/keep");
var _drop = require("./factories/drop");
var _rename = require("./factories/rename");
var _sample = require("./factories/sample");
var _helpers = require("./helpers");
var _walkers = require("./walkers");
var _timeseries = require("./factories/timeseries");
var _rerank = require("./factories/rerank");
var _enrich = require("./factories/enrich");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

class ESQLAstBuilderListener {
  constructor(src) {
    (0, _defineProperty2.default)(this, "ast", []);
    (0, _defineProperty2.default)(this, "inFork", false);
    this.src = src;
  }
  getAst() {
    return {
      ast: this.ast
    };
  }

  /**
   * Exit a parse tree produced by the `showInfo`
   * labeled alternative in `esql_parser.showCommand`.
   * @param ctx the parse tree
   */
  exitShowInfo(ctx) {
    const commandAst = (0, _factories.createCommand)('show', ctx);
    this.ast.push(commandAst);
    commandAst.text = ctx.getText();
    if ((0, _factories.textExistsAndIsValid)(ctx.INFO().getText())) {
      // TODO: these probably should not be functions, instead use "column", like: INFO <identifier>?
      commandAst === null || commandAst === void 0 ? void 0 : commandAst.args.push((0, _factories.createFunction)('info', ctx, (0, _helpers.getPosition)(ctx.INFO().symbol)));
    }
  }

  /**
   * Enter a parse tree produced by `esql_parser.singleStatement`.
   * @param ctx the parse tree
   */
  enterSingleStatement(ctx) {
    this.ast = [];
  }

  /**
   * Exit a parse tree produced by `esql_parser.whereCommand`.
   * @param ctx the parse tree
   */
  exitWhereCommand(ctx) {
    if (this.inFork) {
      return;
    }
    const command = (0, _where.createWhereCommand)(ctx);
    this.ast.push(command);
  }

  /**
   * Exit a parse tree produced by `esql_parser.rowCommand`.
   * @param ctx the parse tree
   */
  exitRowCommand(ctx) {
    const command = (0, _row.createRowCommand)(ctx);
    this.ast.push(command);
  }

  /**
   * Exit a parse tree produced by `esql_parser.fromCommand`.
   * @param ctx the parse tree
   */
  exitFromCommand(ctx) {
    const command = (0, _from.createFromCommand)(ctx);
    this.ast.push(command);
  }

  /**
   * Exit a parse tree produced by `esql_parser.timeseriesCommand`.
   * @param ctx the parse tree
   */
  exitTimeSeriesCommand(ctx) {
    const command = (0, _timeseries.createTimeseriesCommand)(ctx);
    this.ast.push(command);
  }

  /**
   * Exit a parse tree produced by `esql_parser.evalCommand`.
   * @param ctx the parse tree
   */
  exitEvalCommand(ctx) {
    if (this.inFork) {
      return;
    }
    this.ast.push((0, _eval.createEvalCommand)(ctx));
  }

  /**
   * Exit a parse tree produced by `esql_parser.statsCommand`.
   * @param ctx the parse tree
   */
  exitStatsCommand(ctx) {
    if (this.inFork) {
      return;
    }
    const command = (0, _stats.createStatsCommand)(ctx);
    this.ast.push(command);
  }

  /**
   * Exit a parse tree produced by `esql_parser.inlinestatsCommand`.
   * @param ctx the parse tree
   */
  exitInlinestatsCommand(ctx) {
    const command = (0, _factories.createCommand)('inlinestats', ctx);
    this.ast.push(command);

    // STATS expression is optional
    if (ctx._stats) {
      command.args.push(...(0, _walkers.collectAllAggFields)(ctx.aggFields()));
    }
    if (ctx._grouping) {
      command.args.push(...(0, _walkers.visitByOption)(ctx, ctx.fields()));
    }
  }

  /**
   * Exit a parse tree produced by `esql_parser.limitCommand`.
   * @param ctx the parse tree
   */
  exitLimitCommand(ctx) {
    if (this.inFork) {
      return;
    }
    const command = (0, _limit.createLimitCommand)(ctx);
    this.ast.push(command);
  }

  /**
   * Exit a parse tree produced by `esql_parser.sortCommand`.
   * @param ctx the parse tree
   */
  exitSortCommand(ctx) {
    if (this.inFork) {
      return;
    }
    const command = (0, _sort.createSortCommand)(ctx);
    this.ast.push(command);
  }

  /**
   * Exit a parse tree produced by `esql_parser.keepCommand`.
   * @param ctx the parse tree
   */
  exitKeepCommand(ctx) {
    if (this.inFork) {
      return;
    }
    const command = (0, _keep.createKeepCommand)(ctx);
    this.ast.push(command);
  }

  /**
   * Exit a parse tree produced by `esql_parser.dropCommand`.
   * @param ctx the parse tree
   */
  exitDropCommand(ctx) {
    if (this.inFork) {
      return;
    }
    const command = (0, _drop.createDropCommand)(ctx);
    this.ast.push(command);
  }

  /**
   * Exit a parse tree produced by `esql_parser.renameCommand`.
   * @param ctx the parse tree
   */
  exitRenameCommand(ctx) {
    if (this.inFork) {
      return;
    }
    const command = (0, _rename.createRenameCommand)(ctx);
    this.ast.push(command);
  }

  /**
   * Exit a parse tree produced by `esql_parser.dissectCommand`.
   * @param ctx the parse tree
   */
  exitDissectCommand(ctx) {
    if (this.inFork) {
      return;
    }
    this.ast.push((0, _dissect.createDissectCommand)(ctx));
  }

  /**
   * Exit a parse tree produced by `esql_parser.grokCommand`.
   * @param ctx the parse tree
   */
  exitGrokCommand(ctx) {
    if (this.inFork) {
      return;
    }
    const command = (0, _grok.createGrokCommand)(ctx);
    this.ast.push(command);
  }

  /**
   * Exit a parse tree produced by `esql_parser.mvExpandCommand`.
   * @param ctx the parse tree
   */
  exitMvExpandCommand(ctx) {
    if (this.inFork) {
      return;
    }
    const command = (0, _mv_expand.createMvExpandCommand)(ctx);
    this.ast.push(command);
  }

  /**
   * Enter a parse tree produced by `esql_parser.showCommand`.
   * @param ctx the parse tree
   */
  enterShowCommand(ctx) {
    const command = (0, _factories.createCommand)('show', ctx);
    this.ast.push(command);
  }

  /**
   * Exit a parse tree produced by `esql_parser.enrichCommand`.
   * @param ctx the parse tree
   */
  exitEnrichCommand(ctx) {
    if (this.inFork) {
      return;
    }
    const command = (0, _enrich.createEnrichCommand)(ctx);
    this.ast.push(command);
  }

  /**
   * Exit a parse tree produced by `esql_parser.joinCommand`.
   *
   * Parse the JOIN command:
   *
   * ```
   * <type> JOIN identifier [ AS identifier ] ON expression [, expression [, ... ]]
   * ```
   *
   * @param ctx the parse tree
   */
  exitJoinCommand(ctx) {
    if (this.inFork) {
      return;
    }
    const command = (0, _join.createJoinCommand)(ctx);
    this.ast.push(command);
  }
  enterForkCommand() {
    this.inFork = true;
  }

  /**
   * NOTE — every new command supported in fork needs to be added
   * to createForkCommand!
   */
  exitForkCommand(ctx) {
    const command = (0, _fork.createForkCommand)(ctx);
    this.ast.push(command);
    this.inFork = false;
  }

  /**
   * Exit a parse tree produced by `esql_parser.changePointCommand`.
   *
   * Parse the CHANGE_POINT command:
   *
   * CHANGE_POINT <value> [ ON <key> ] [ AS <target-type>, <target-pvalue> ]
   *
   * @param ctx the parse tree
   */
  exitChangePointCommand(ctx) {
    if (this.inFork) {
      return;
    }
    const command = (0, _change_point.createChangePointCommand)(ctx);
    this.ast.push(command);
  }

  /**
   * Exit a parse tree produced by `esql_parser.rerankCommand`.
   *
   * Parse the RERANK command:
   *
   * RERANK <query> ON <fields> WITH <referenceId>
   *
   * @param ctx the parse tree
   */
  exitRerankCommand(ctx) {
    const command = (0, _rerank.createRerankCommand)(ctx);
    this.ast.push(command);
  }

  /**
   * Exit a parse tree produced by `esql_parser.completionCommand`.
   *
   * Parse the COMPLETION command:
   *
   * COMPLETION <prompt> WITH <inferenceId> [ AS <targetField> ]
   *
   * @param ctx the parse tree
   */
  exitCompletionCommand(ctx) {
    if (this.inFork) {
      return;
    }
    const command = (0, _completion.createCompletionCommand)(ctx);
    this.ast.push(command);
  }
  exitSampleCommand(ctx) {
    if (this.inFork) {
      return;
    }
    const command = (0, _sample.createSampleCommand)(ctx);
    this.ast.push(command);
  }

  /**
   * Exit a parse tree produced by `esql_parser.rrfCommand`.
   *
   * Parse the RRF (Reciprocal Rank Fusion) command:
   *
   * RRF
   *
   * @param ctx the parse tree
   */
  exitRrfCommand(ctx) {
    const command = (0, _factories.createCommand)('rrf', ctx);
    this.ast.push(command);
  }
  enterEveryRule(ctx) {
    // method not implemented, added to satisfy interface expectation
  }
  visitErrorNode(node) {
    // method not implemented, added to satisfy interface expectation
  }
  visitTerminal(node) {
    // method not implemented, added to satisfy interface expectation
  }
  exitEveryRule(ctx) {
    // method not implemented, added to satisfy interface expectation
  }
}
exports.ESQLAstBuilderListener = ESQLAstBuilderListener;