"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MappedColors = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = _interopRequireDefault(require("lodash"));
var _color = _interopRequireDefault(require("color"));
var _common = require("../../../common");
var _colors = require("../../static/colors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const standardizeColor = color => new _color.default(color).hex().toLowerCase();

/**
 * Maintains a lookup table that associates the value (key) with a hex color (value)
 * across the visualizations.
 * Provides functions to interact with the lookup table
 */
class MappedColors {
  constructor(uiSettings, colorPaletteFn = _colors.createColorPalette) {
    (0, _defineProperty2.default)(this, "_oldMap", void 0);
    (0, _defineProperty2.default)(this, "_mapping", void 0);
    this.uiSettings = uiSettings;
    this.colorPaletteFn = colorPaletteFn;
    this._oldMap = {};
    this._mapping = {};
  }
  getConfigColorMapping() {
    var _this$uiSettings;
    return _lodash.default.mapValues(((_this$uiSettings = this.uiSettings) === null || _this$uiSettings === void 0 ? void 0 : _this$uiSettings.get(_common.COLOR_MAPPING_SETTING)) || {}, standardizeColor);
  }
  get oldMap() {
    return this._oldMap;
  }
  get mapping() {
    return this._mapping;
  }
  get(key) {
    return this.getConfigColorMapping()[key] || this._mapping[key];
  }
  getColorFromConfig(key) {
    return this.getConfigColorMapping()[key];
  }
  flush() {
    this._oldMap = _lodash.default.clone(this._mapping);
    this._mapping = {};
  }
  purge() {
    this._oldMap = {};
    this._mapping = {};
  }
  mapKeys(keys) {
    const configMapping = this.getConfigColorMapping();
    const configColors = _lodash.default.values(configMapping);
    const oldColors = _lodash.default.values(this._oldMap);
    const keysToMap = [];
    _lodash.default.each(keys, key => {
      // If this key is mapped in the config, it's unnecessary to have it mapped here
      if (configMapping[key]) delete this._mapping[key];

      // If this key is mapped to a color used by the config color mapping, we need to remap it
      if (_lodash.default.includes(configColors, this._mapping[key])) keysToMap.push(key);

      // if key exist in oldMap, move it to mapping
      if (this._oldMap[key]) this._mapping[key] = this._oldMap[key];

      // If this key isn't mapped, we need to map it
      if (this.get(key) == null) keysToMap.push(key);
    });

    // Generate a color palette big enough that all new keys can have unique color values
    const allColors = (0, _lodash.default)(this._mapping).values().union(configColors).union(oldColors).value();
    const colorPalette = this.colorPaletteFn(allColors.length + keysToMap.length);
    let newColors = _lodash.default.difference(colorPalette, allColors);
    while (keysToMap.length > newColors.length) {
      newColors = newColors.concat(_lodash.default.sampleSize(allColors, keysToMap.length - newColors.length));
    }
    _lodash.default.merge(this._mapping, _lodash.default.zipObject(keysToMap, newColors));
  }
}
exports.MappedColors = MappedColors;