"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.withSpaceSolutionDisabledFeatures = withSpaceSolutionDisabledFeatures;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getFeatureIdsForCategories = (features, categories) => {
  return features.filter(feature =>
  // We need to make sure we only reference non-deprecated features
  feature.category && !feature.deprecated ? categories.includes(feature.category.id) : false).map(feature => feature.id);
};

/**
 * These features will be enabled per solution view, even if they fall under a category that is disabled in the solution.
 */

const enabledFeaturesPerSolution = {
  es: ['observabilityAIAssistant'],
  oblt: [],
  security: []
};

/**
 * When a space has a `solution` defined, we want to disable features that are not part of that solution.
 * This function takes the current space's disabled features and the space solution and returns
 * the updated array of disabled features.
 *
 * @param features The list of all Kibana registered features.
 * @param spaceDisabledFeatures The current space's disabled features
 * @param spaceSolution The current space's solution (es, oblt, security or classic)
 * @returns The updated array of disabled features
 */
function withSpaceSolutionDisabledFeatures(features, spaceDisabledFeatures = [], spaceSolution = 'classic') {
  if (spaceSolution === 'classic') {
    return spaceDisabledFeatures;
  }
  let disabledFeatureKeysFromSolution = [];
  if (spaceSolution === 'es') {
    disabledFeatureKeysFromSolution = getFeatureIdsForCategories(features, ['observability', 'securitySolution']).filter(featureId => !enabledFeaturesPerSolution.es.includes(featureId));
  } else if (spaceSolution === 'oblt') {
    disabledFeatureKeysFromSolution = getFeatureIdsForCategories(features, ['securitySolution']).filter(featureId => !enabledFeaturesPerSolution.oblt.includes(featureId));
  } else if (spaceSolution === 'security') {
    disabledFeatureKeysFromSolution = getFeatureIdsForCategories(features, ['observability']).filter(featureId => !enabledFeaturesPerSolution.security.includes(featureId));
  }
  return Array.from(new Set([...disabledFeatureKeysFromSolution]));
}