"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DefaultSummarySearchClient = void 0;
var _esQuery = require("@kbn/es-query");
var _sloSchema = require("@kbn/slo-schema");
var _std = require("@kbn/std");
var _lodash = require("lodash");
var _constants = require("../../../common/constants");
var _number = require("../../utils/number");
var _queries = require("../../utils/queries");
var _slo_settings = require("../slo_settings");
var _transform_generators = require("../transform_generators");
var _remote_summary_doc_to_slo = require("../unsafe_federated/remote_summary_doc_to_slo");
var _utils = require("../utils");
var _types = require("./types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class DefaultSummarySearchClient {
  constructor(esClient, soClient, logger, spaceId) {
    this.esClient = esClient;
    this.soClient = soClient;
    this.logger = logger;
    this.spaceId = spaceId;
  }
  async search(kqlQuery, filters, sort, pagination, hideStale) {
    var _parsedFilters$filter, _parsedFilters$must_n;
    const parsedFilters = (0, _transform_generators.parseStringFilters)(filters, this.logger);
    const settings = await (0, _slo_settings.getSloSettings)(this.soClient);
    const {
      indices
    } = await (0, _slo_settings.getSummaryIndices)(this.esClient, settings);
    const esParams = (0, _queries.createEsParams)({
      index: indices,
      track_total_hits: true,
      query: {
        bool: {
          filter: [{
            term: {
              spaceId: this.spaceId
            }
          }, ...excludeStaleSummaryFilter(settings, kqlQuery, hideStale), (0, _transform_generators.getElasticsearchQueryOrThrow)(kqlQuery), ...((_parsedFilters$filter = parsedFilters.filter) !== null && _parsedFilters$filter !== void 0 ? _parsedFilters$filter : [])],
          must_not: [...((_parsedFilters$must_n = parsedFilters.must_not) !== null && _parsedFilters$must_n !== void 0 ? _parsedFilters$must_n : [])]
        }
      },
      sort: {
        // non-temp first, then temp documents
        isTempDoc: {
          order: 'asc'
        },
        [toDocumentSortField(sort.field)]: {
          order: sort.direction
        },
        'slo.id': {
          order: 'asc'
        },
        'slo.instanceId': {
          order: 'asc'
        }
      },
      ...toPaginationQuery(pagination)
    });
    try {
      var _summarySearch$hits$t;
      const summarySearch = await (0, _queries.typedSearch)(this.esClient, esParams);
      const total = (_summarySearch$hits$t = summarySearch.hits.total.value) !== null && _summarySearch$hits$t !== void 0 ? _summarySearch$hits$t : 0;
      if (total === 0) {
        return {
          total: 0,
          ...pagination,
          results: []
        };
      }
      const [tempSummaryDocuments, summaryDocuments] = (0, _lodash.partition)(summarySearch.hits.hits, doc => {
        var _doc$_source;
        return !!((_doc$_source = doc._source) !== null && _doc$_source !== void 0 && _doc$_source.isTempDoc);
      });

      // TODO filter out remote summary documents from the deletion of outdated summaries
      const summarySloIds = summaryDocuments.map(doc => doc._source.slo.id);
      await this.deleteOutdatedTemporarySummaries(summarySloIds);
      const tempSummaryDocumentsDeduped = tempSummaryDocuments.filter(doc => !summarySloIds.includes(doc._source.slo.id));
      const finalResults = summaryDocuments.concat(tempSummaryDocumentsDeduped).slice(0, (0, _types.isCursorPagination)(pagination) ? pagination.size : pagination.perPage);
      const finalTotal = total - (tempSummaryDocuments.length - tempSummaryDocumentsDeduped.length);
      const paginationResults = (0, _types.isCursorPagination)(pagination) ? {
        searchAfter: finalResults[finalResults.length - 1].sort,
        size: pagination.size
      } : pagination;
      return {
        ...paginationResults,
        total: finalTotal,
        results: finalResults.map(doc => {
          var _summaryDoc$kibanaUrl, _summaryDoc$slo$insta, _summaryDoc$fiveMinut, _summaryDoc$fiveMinut2, _summaryDoc$oneHourBu, _summaryDoc$oneHourBu2, _summaryDoc$oneDayBur, _summaryDoc$oneDayBur2;
          const summaryDoc = doc._source;
          const remoteName = getRemoteClusterName(doc._index);
          const isRemote = !!remoteName;
          let remoteSloDefinition;
          if (isRemote) {
            remoteSloDefinition = (0, _remote_summary_doc_to_slo.fromRemoteSummaryDocumentToSloDefinition)(summaryDoc, this.logger);
          }
          return {
            ...(isRemote && !!remoteSloDefinition && {
              remote: {
                kibanaUrl: (_summaryDoc$kibanaUrl = summaryDoc.kibanaUrl) !== null && _summaryDoc$kibanaUrl !== void 0 ? _summaryDoc$kibanaUrl : '',
                remoteName,
                slo: remoteSloDefinition
              }
            }),
            sloId: summaryDoc.slo.id,
            instanceId: (_summaryDoc$slo$insta = summaryDoc.slo.instanceId) !== null && _summaryDoc$slo$insta !== void 0 ? _summaryDoc$slo$insta : _sloSchema.ALL_VALUE,
            summary: {
              errorBudget: {
                initial: (0, _number.toHighPrecision)(summaryDoc.errorBudgetInitial),
                consumed: (0, _number.toHighPrecision)(summaryDoc.errorBudgetConsumed),
                remaining: (0, _number.toHighPrecision)(summaryDoc.errorBudgetRemaining),
                isEstimated: summaryDoc.errorBudgetEstimated
              },
              sliValue: (0, _number.toHighPrecision)(doc._source.sliValue),
              status: summaryDoc.status,
              summaryUpdatedAt: summaryDoc.summaryUpdatedAt,
              fiveMinuteBurnRate: (0, _number.toHighPrecision)((_summaryDoc$fiveMinut = (_summaryDoc$fiveMinut2 = summaryDoc.fiveMinuteBurnRate) === null || _summaryDoc$fiveMinut2 === void 0 ? void 0 : _summaryDoc$fiveMinut2.value) !== null && _summaryDoc$fiveMinut !== void 0 ? _summaryDoc$fiveMinut : 0),
              oneHourBurnRate: (0, _number.toHighPrecision)((_summaryDoc$oneHourBu = (_summaryDoc$oneHourBu2 = summaryDoc.oneHourBurnRate) === null || _summaryDoc$oneHourBu2 === void 0 ? void 0 : _summaryDoc$oneHourBu2.value) !== null && _summaryDoc$oneHourBu !== void 0 ? _summaryDoc$oneHourBu : 0),
              oneDayBurnRate: (0, _number.toHighPrecision)((_summaryDoc$oneDayBur = (_summaryDoc$oneDayBur2 = summaryDoc.oneDayBurnRate) === null || _summaryDoc$oneDayBur2 === void 0 ? void 0 : _summaryDoc$oneDayBur2.value) !== null && _summaryDoc$oneDayBur !== void 0 ? _summaryDoc$oneDayBur : 0)
            },
            groupings: (0, _utils.getFlattenedGroupings)({
              groupings: summaryDoc.slo.groupings,
              groupBy: summaryDoc.slo.groupBy
            })
          };
        })
      };
    } catch (err) {
      this.logger.debug(`Error while searching SLO summary documents. ${err}`);
      return {
        total: 0,
        ...pagination,
        results: []
      };
    }
  }
  async deleteOutdatedTemporarySummaries(summarySloIds) {
    // Always attempt to delete temporary summary documents with an existing non-temp summary document
    // The temp summary documents are _eventually_ removed as we get through the real summary documents

    await this.esClient.deleteByQuery({
      index: _constants.SUMMARY_DESTINATION_INDEX_PATTERN,
      wait_for_completion: false,
      conflicts: 'proceed',
      slices: 'auto',
      query: {
        bool: {
          filter: [{
            terms: {
              'slo.id': summarySloIds
            }
          }, {
            term: {
              isTempDoc: true
            }
          }]
        }
      }
    });
  }
}
exports.DefaultSummarySearchClient = DefaultSummarySearchClient;
function excludeStaleSummaryFilter(settings, kqlFilter, hideStale) {
  if (kqlFilter.includes('summaryUpdatedAt') || !settings.staleThresholdInHours || !hideStale) {
    return [];
  }
  return [{
    bool: {
      should: [{
        term: {
          isTempDoc: true
        }
      }, {
        range: {
          summaryUpdatedAt: {
            gte: `now-${settings.staleThresholdInHours}h`
          }
        }
      }]
    }
  }];
}
function getRemoteClusterName(index) {
  if ((0, _esQuery.isCCSRemoteIndexName)(index)) {
    return index.substring(0, index.indexOf(':'));
  }
}
function toDocumentSortField(field) {
  switch (field) {
    case 'error_budget_consumed':
      return 'errorBudgetConsumed';
    case 'error_budget_remaining':
      return 'errorBudgetRemaining';
    case 'status':
      return 'status';
    case 'sli_value':
      return 'sliValue';
    case 'burn_rate_5m':
      return 'fiveMinuteBurnRate.value';
    case 'burn_rate_1h':
      return 'oneHourBurnRate.value';
    case 'burn_rate_1d':
      return 'oneDayBurnRate.value';
    default:
      (0, _std.assertNever)(field);
  }
}
function toPaginationQuery(pagination) {
  if ((0, _types.isCursorPagination)(pagination)) {
    return {
      size: pagination.size * 2,
      // Potential duplicates between temp and non-temp summaries
      search_after: pagination.searchAfter
    };
  }
  return {
    size: pagination.perPage * 2,
    // Potential duplicates between temp and non-temp summaries
    from: (pagination.page - 1) * pagination.perPage
  };
}