"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getSignalsQueryMapFromThreatIndex = getSignalsQueryMapFromThreatIndex;
var _lodash = require("lodash");
var _types = require("./types");
var _get_threat_list = require("./get_threat_list");
var _utils = require("./utils");
var _enrich_signal_threat_matches = require("./enrich_signal_threat_matches");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * fetches threats and creates signals map from results, that matches signal is with list of threat queries
 */
/**
 * fetches threats and creates signals map from results, that matches signal is with list of threat queries
 * @param options.termsQueryAllowed - if terms query allowed to be executed, then signalValueMap should be provided
 * @param options.signalValueMap - map of signal values from terms query results
 */
async function getSignalsQueryMapFromThreatIndex(options) {
  const {
    threatSearchParams,
    eventsCount,
    termsQueryAllowed
  } = options;
  let threatList;
  const signalsQueryMap = new Map();
  // number of threat matches per signal is limited by MAX_NUMBER_OF_SIGNAL_MATCHES. Once it hits this number, threats stop to be processed for a signal
  const maxThreatsReachedMap = new Map();
  const addSignalValueToMap = ({
    signalId,
    threatHit,
    decodedQuery
  }) => {
    const signalMatch = signalsQueryMap.get(signalId);
    const threatQuery = {
      // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
      id: threatHit._id,
      index: threatHit._index,
      field: decodedQuery.field,
      value: decodedQuery.value,
      queryType: decodedQuery.queryType
    };
    if (!signalMatch) {
      signalsQueryMap.set(signalId, [threatQuery]);
      return;
    }
    if (signalMatch.length === _enrich_signal_threat_matches.MAX_NUMBER_OF_SIGNAL_MATCHES) {
      maxThreatsReachedMap.set(signalId, true);
    } else if (signalMatch.length < _enrich_signal_threat_matches.MAX_NUMBER_OF_SIGNAL_MATCHES) {
      signalMatch.push(threatQuery);
    }
  };
  threatList = await (0, _get_threat_list.getThreatList)({
    ...threatSearchParams,
    searchAfter: undefined
  });
  while (maxThreatsReachedMap.size < eventsCount && ((_threatList = threatList) === null || _threatList === void 0 ? void 0 : _threatList.hits.hits.length) > 0) {
    var _threatList;
    threatList.hits.hits.forEach(threatHit => {
      const matchedQueries = Array.isArray(threatHit === null || threatHit === void 0 ? void 0 : threatHit.matched_queries) ? threatHit.matched_queries : [];
      matchedQueries.forEach(matchedQuery => {
        const decodedQuery = (0, _utils.decodeThreatMatchNamedQuery)(matchedQuery);
        const signalId = decodedQuery.id;
        if (decodedQuery.queryType === _types.ThreatMatchQueryType.term && termsQueryAllowed) {
          const threatValue = (0, _lodash.get)(threatHit === null || threatHit === void 0 ? void 0 : threatHit._source, decodedQuery.value);
          const values = Array.isArray(threatValue) ? threatValue : [threatValue];
          values.forEach(value => {
            if (value && options.signalValueMap) {
              const ids = options.signalValueMap[decodedQuery.field][value === null || value === void 0 ? void 0 : value.toString()];
              ids === null || ids === void 0 ? void 0 : ids.forEach(id => {
                addSignalValueToMap({
                  signalId: id,
                  threatHit,
                  decodedQuery
                });
              });
            }
          });
        } else {
          if (!signalId) {
            return;
          }
          addSignalValueToMap({
            signalId,
            threatHit,
            decodedQuery
          });
        }
      });
    });
    threatList = await (0, _get_threat_list.getThreatList)({
      ...threatSearchParams,
      searchAfter: threatList.hits.hits[threatList.hits.hits.length - 1].sort
    });
  }
  return signalsQueryMap;
}