"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.bulkEditRules = void 0;
var _convert_alerting_rule_to_rule_response = require("../detection_rules_client/converters/convert_alerting_rule_to_rule_response");
var _calculate_is_customized = require("../detection_rules_client/mergers/rule_source/calculate_is_customized");
var _action_to_rules_client_operation = require("./action_to_rules_client_operation");
var _rule_params_modifier = require("./rule_params_modifier");
var _split_bulk_edit_actions = require("./split_bulk_edit_actions");
var _validations = require("./validations");
var _invariant = require("../../../../../../common/utils/invariant");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * calls rulesClient.bulkEdit
 * transforms bulk actions payload into rulesClient compatible operations
 * enriches query filter with rule types search queries
 * @param BulkEditRulesArguments
 * @returns edited rules and caught errors
 */
const bulkEditRules = async ({
  actionsClient,
  rulesClient,
  prebuiltRuleAssetClient,
  rules,
  actions,
  mlAuthz,
  ruleCustomizationStatus
}) => {
  // Split operations
  const {
    attributesActions,
    paramsActions
  } = (0, _split_bulk_edit_actions.splitBulkEditActions)(actions);
  const operations = attributesActions.map(attribute => (0, _action_to_rules_client_operation.bulkEditActionToRulesClientOperation)(actionsClient, attribute)).flat();

  // Check if there are any prebuilt rules and fetch their base versions
  const prebuiltRules = rules.filter(rule => rule.params.immutable);
  const baseVersions = await prebuiltRuleAssetClient.fetchAssetsByVersion(prebuiltRules.map(rule => ({
    rule_id: rule.params.ruleId,
    version: rule.params.version
  })));
  const baseVersionsMap = new Map(baseVersions.map(baseVersion => [baseVersion.rule_id, baseVersion]));
  const currentRulesMap = new Map(rules.map(rule => [rule.id, rule]));
  const result = await rulesClient.bulkEdit({
    ids: Array.from(currentRulesMap.keys()),
    operations,
    // Rules Client applies operations to rules client aware fields like tags
    // the rule before passing it to paramsModifier().
    paramsModifier: async partiallyModifiedRule => {
      const ruleParams = partiallyModifiedRule.params;
      await (0, _validations.validateBulkEditRule)({
        mlAuthz,
        ruleType: ruleParams.type,
        edit: actions,
        immutable: ruleParams.immutable,
        ruleCustomizationStatus
      });
      const currentRule = currentRulesMap.get(partiallyModifiedRule.id);
      (0, _invariant.invariant)(currentRule, "Unable to extract rule's current data in paramsModifier");
      const {
        modifiedParams,
        isParamsUpdateSkipped
      } = (0, _rule_params_modifier.ruleParamsModifier)(ruleParams, paramsActions);
      const nextRule = (0, _convert_alerting_rule_to_rule_response.convertAlertingRuleToRuleResponse)({
        ...partiallyModifiedRule,
        params: modifiedParams
      });
      let isCustomized = false;
      if (nextRule.immutable === true) {
        isCustomized = (0, _calculate_is_customized.calculateIsCustomized)({
          baseRule: baseVersionsMap.get(nextRule.rule_id),
          currentRule: (0, _convert_alerting_rule_to_rule_response.convertAlertingRuleToRuleResponse)(currentRule),
          nextRule
        });
      }
      const ruleSource = nextRule.immutable === true ? {
        type: 'external',
        isCustomized
      } : {
        type: 'internal'
      };
      modifiedParams.ruleSource = ruleSource;
      return {
        modifiedParams,
        isParamsUpdateSkipped
      };
    }
  });
  return result;
};
exports.bulkEditRules = bulkEditRules;