"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.setAlertTagsRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _fp = require("lodash/fp");
var _zodHelpers = require("@kbn/zod-helpers");
var _alert_tags = require("../../../../../common/api/detection_engine/alert_tags");
var _constants = require("../../../../../common/constants");
var _utils = require("../utils");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const setAlertTagsRoute = router => {
  router.versioned.post({
    path: _constants.DETECTION_ENGINE_ALERT_TAGS_URL,
    access: 'public',
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: {
        body: (0, _zodHelpers.buildRouteValidationWithZod)(_alert_tags.SetAlertTagsRequestBody)
      }
    }
  }, async (context, request, response) => {
    var _securitySolution$get;
    const {
      tags,
      ids
    } = request.body;
    const core = await context.core;
    const securitySolution = await context.securitySolution;
    const esClient = core.elasticsearch.client.asCurrentUser;
    const siemClient = securitySolution === null || securitySolution === void 0 ? void 0 : securitySolution.getAppClient();
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    const validationErrors = (0, _helpers.validateAlertTagsArrays)(tags, ids);
    const spaceId = (_securitySolution$get = securitySolution === null || securitySolution === void 0 ? void 0 : securitySolution.getSpaceId()) !== null && _securitySolution$get !== void 0 ? _securitySolution$get : 'default';
    if (validationErrors.length) {
      return siemResponse.error({
        statusCode: 400,
        body: validationErrors
      });
    }
    if (!siemClient) {
      return siemResponse.error({
        statusCode: 404
      });
    }
    const tagsToAdd = (0, _fp.uniq)(tags.tags_to_add);
    const tagsToRemove = (0, _fp.uniq)(tags.tags_to_remove);
    const painlessScript = {
      params: {
        tagsToAdd,
        tagsToRemove
      },
      source: `List newTagsArray = [];
        if (ctx._source["kibana.alert.workflow_tags"] != null) {
          for (tag in ctx._source["kibana.alert.workflow_tags"]) {
            if (!params.tagsToRemove.contains(tag)) {
              newTagsArray.add(tag);
            }
          }
          for (tag in params.tagsToAdd) {
            if (!newTagsArray.contains(tag)) {
              newTagsArray.add(tag)
            }
          }
          ctx._source["kibana.alert.workflow_tags"] = newTagsArray;
        } else {
          ctx._source["kibana.alert.workflow_tags"] = params.tagsToAdd;
        }
        `,
      lang: 'painless'
    };
    const bulkUpdateRequest = [];
    for (const id of ids) {
      bulkUpdateRequest.push({
        update: {
          _index: `${_constants.DEFAULT_ALERTS_INDEX}-${spaceId}`,
          _id: id
        }
      }, {
        script: painlessScript
      });
    }
    try {
      const body = await esClient.updateByQuery({
        index: `${_constants.DEFAULT_ALERTS_INDEX}-${spaceId}`,
        refresh: true,
        body: {
          script: painlessScript,
          query: {
            bool: {
              filter: {
                terms: {
                  _id: ids
                }
              }
            }
          }
        }
      });
      return response.ok({
        body
      });
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return siemResponse.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.setAlertTagsRoute = setAlertTagsRoute;