"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createPackRoute = void 0;
var _momentTimezone = _interopRequireDefault(require("moment-timezone"));
var _saferLodashSet = require("@kbn/safer-lodash-set");
var _lodash = require("lodash");
var _immer = require("immer");
var _common = require("@kbn/fleet-plugin/common");
var _route_validation = require("../../utils/build_validation/route_validation");
var _constants = require("../../../common/constants");
var _common2 = require("../../../common");
var _types = require("../../../common/types");
var _utils = require("./utils");
var _utils2 = require("../utils");
var _api = require("../../../common/api");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createPackRoute = (router, osqueryContext) => {
  router.versioned.post({
    access: 'public',
    path: '/api/osquery/packs',
    security: {
      authz: {
        requiredPrivileges: [`${_common2.PLUGIN_ID}-writePacks`]
      }
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: {
        body: (0, _route_validation.buildRouteValidation)(_api.createPackRequestBodySchema)
      }
    }
  }, async (context, request, response) => {
    var _coreContext$security, _await$packagePolicyS;
    const coreContext = await context.core;
    const esClient = coreContext.elasticsearch.client.asCurrentUser;
    const savedObjectsClient = coreContext.savedObjects.client;
    const internalSavedObjectsClient = await (0, _utils2.getInternalSavedObjectsClient)(osqueryContext.getStartServices);
    const agentPolicyService = osqueryContext.service.getAgentPolicyService();
    const packagePolicyService = osqueryContext.service.getPackagePolicyService();
    const currentUser = (_coreContext$security = coreContext.security.authc.getCurrentUser()) === null || _coreContext$security === void 0 ? void 0 : _coreContext$security.username;

    // eslint-disable-next-line @typescript-eslint/naming-convention
    const {
      name,
      description,
      queries,
      enabled,
      policy_ids,
      shards = {}
    } = request.body;
    const conflictingEntries = await savedObjectsClient.find({
      type: _types.packSavedObjectType,
      filter: `${_types.packSavedObjectType}.attributes.name: "${name}"`
    });
    if (conflictingEntries.saved_objects.length && (0, _lodash.some)(conflictingEntries.saved_objects, ['attributes.name', name])) {
      return response.conflict({
        body: `Pack with name "${name}" already exists.`
      });
    }
    const {
      items: packagePolicies
    } = (_await$packagePolicyS = await (packagePolicyService === null || packagePolicyService === void 0 ? void 0 : packagePolicyService.list(internalSavedObjectsClient, {
      kuery: `${_common.PACKAGE_POLICY_SAVED_OBJECT_TYPE}.package.name:${_common2.OSQUERY_INTEGRATION_NAME}`,
      perPage: 1000,
      page: 1
    }))) !== null && _await$packagePolicyS !== void 0 ? _await$packagePolicyS : {
      items: []
    };
    const {
      policiesList,
      invalidPolicies
    } = (0, _utils.getInitialPolicies)(packagePolicies, policy_ids, shards);
    if (invalidPolicies !== null && invalidPolicies !== void 0 && invalidPolicies.length) {
      return response.badRequest({
        body: `The following policy ids are invalid: ${invalidPolicies.join(', ')}`
      });
    }
    const agentPolicies = await (agentPolicyService === null || agentPolicyService === void 0 ? void 0 : agentPolicyService.getByIds(internalSavedObjectsClient, policiesList));
    const policyShards = (0, _utils.findMatchingShards)(agentPolicies, shards);
    const agentPoliciesIdMap = (0, _lodash.mapKeys)(agentPolicies, 'id');
    const references = policiesList.map(id => {
      var _agentPoliciesIdMap$i;
      return {
        id,
        name: (_agentPoliciesIdMap$i = agentPoliciesIdMap[id]) === null || _agentPoliciesIdMap$i === void 0 ? void 0 : _agentPoliciesIdMap$i.name,
        type: _common.LEGACY_AGENT_POLICY_SAVED_OBJECT_TYPE
      };
    });
    const packSO = await savedObjectsClient.create(_types.packSavedObjectType, {
      name,
      description,
      queries: (0, _utils.convertPackQueriesToSO)(queries),
      enabled,
      created_at: (0, _momentTimezone.default)().toISOString(),
      created_by: currentUser,
      updated_at: (0, _momentTimezone.default)().toISOString(),
      updated_by: currentUser,
      shards: (0, _utils2.convertShardsToArray)(shards)
    }, {
      references,
      refresh: 'wait_for'
    });
    if (enabled && policiesList.length) {
      await Promise.all(policiesList.map(agentPolicyId => {
        const packagePolicy = packagePolicies.find(policy => policy.policy_ids.includes(agentPolicyId));
        if (packagePolicy) {
          return packagePolicyService === null || packagePolicyService === void 0 ? void 0 : packagePolicyService.update(internalSavedObjectsClient, esClient, packagePolicy.id, (0, _immer.produce)(packagePolicy, draft => {
            var _policyShards$agentPo;
            (0, _lodash.unset)(draft, 'id');
            if (!(0, _lodash.has)(draft, 'inputs[0].streams')) {
              (0, _saferLodashSet.set)(draft, 'inputs[0].streams', []);
            }
            (0, _saferLodashSet.set)(draft, `inputs[0].config.osquery.value.packs.${packSO.attributes.name}`, {
              shard: (_policyShards$agentPo = policyShards[agentPolicyId]) !== null && _policyShards$agentPo !== void 0 ? _policyShards$agentPo : 100,
              queries: (0, _utils.convertSOQueriesToPackConfig)(queries)
            });
            return draft;
          }));
        }
      }));
    }
    (0, _saferLodashSet.set)(packSO, 'attributes.queries', queries);
    const {
      attributes
    } = packSO;
    const data = {
      name: attributes.name,
      description: attributes.description,
      queries: attributes.queries,
      version: attributes.version,
      enabled: attributes.enabled,
      created_at: attributes.created_at,
      created_by: attributes.created_by,
      updated_at: attributes.updated_at,
      updated_by: attributes.updated_by,
      policy_ids: attributes.policy_ids,
      shards: attributes.shards,
      saved_object_id: packSO.id
    };
    return response.ok({
      body: {
        data
      }
    });
  });
};
exports.createPackRoute = createPackRoute;