"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RETRIEVE_DOCUMENTATION_NAME = void 0;
exports.registerDocumentationFunction = registerDocumentationFunction;
var _productDocCommon = require("@kbn/product-doc-common");
var _common = require("@kbn/observability-ai-assistant-plugin/common");
var _inferenceCommon = require("@kbn/inference-common");
var _server = require("@kbn/observability-ai-assistant-plugin/server");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const RETRIEVE_DOCUMENTATION_NAME = exports.RETRIEVE_DOCUMENTATION_NAME = 'retrieve_elastic_doc';
async function registerDocumentationFunction({
  functions,
  resources,
  pluginsStart: {
    llmTasks
  }
}) {
  var _await$getInferenceId, _await$llmTasks$retri;
  const esClient = (await resources.context.core).elasticsearch.client;
  const inferenceId = (_await$getInferenceId = await (0, _server.getInferenceIdFromWriteIndex)(esClient)) !== null && _await$getInferenceId !== void 0 ? _await$getInferenceId : _inferenceCommon.defaultInferenceEndpoints.ELSER;
  const isProductDocAvailable = (_await$llmTasks$retri = await llmTasks.retrieveDocumentationAvailable({
    inferenceId
  })) !== null && _await$llmTasks$retri !== void 0 ? _await$llmTasks$retri : false;
  if (isProductDocAvailable) {
    functions.registerInstruction(({
      availableFunctionNames
    }) => {
      return availableFunctionNames.includes(RETRIEVE_DOCUMENTATION_NAME) ? `When asked questions about the Elastic stack or products, You should use the ${RETRIEVE_DOCUMENTATION_NAME} function before answering,
      to retrieve documentation related to the question. Consider that the documentation returned by the function
      is always more up to date and accurate than any own internal knowledge you might have.` : undefined;
    });
  }
  functions.registerFunction({
    name: RETRIEVE_DOCUMENTATION_NAME,
    visibility: isProductDocAvailable ? _common.FunctionVisibility.All : _common.FunctionVisibility.Internal,
    description: `Use this function to retrieve documentation about Elastic products.
      You can retrieve documentation about the Elastic stack, such as Kibana and Elasticsearch,
      or for Elastic solutions, such as Elastic Security, Elastic Observability or Elastic Enterprise Search
      `,
    parameters: {
      type: 'object',
      properties: {
        query: {
          description: `The query to use to retrieve documentation
            Always write the query in English, as the documentation is available only in English.
            Examples:
            - "How to enable TLS for Elasticsearch?"
            - "What is Kibana Lens?"`,
          type: 'string'
        },
        product: {
          description: `If specified, will filter the products to retrieve documentation for
            Possible options are:
            - "kibana": Kibana product
            - "elasticsearch": Elasticsearch product
            - "observability": Elastic Observability solution
            - "security": Elastic Security solution
            If not specified, will search against all products
            `,
          type: 'string',
          enum: Object.values(_productDocCommon.DocumentationProduct)
        }
      },
      required: ['query']
    }
  }, async ({
    arguments: {
      query,
      product
    },
    connectorId,
    simulateFunctionCalling
  }) => {
    const response = await llmTasks.retrieveDocumentation({
      searchTerm: query,
      products: product ? [product] : undefined,
      max: 3,
      connectorId,
      request: resources.request,
      functionCalling: simulateFunctionCalling ? 'simulated' : 'auto',
      inferenceId
    });
    return {
      content: {
        documents: response.documents
      }
    };
  });
}